<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\BinLocationStockSnapshot;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonCommon\Controllers\ViisonCommonBaseController;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\BinLocationStockChange;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\RelocationStockChangeList;
use Shopware\Plugins\ViisonPickwareERP\Components\Warehouse\WarehouseIntegrity;

/**
 * Backend controller for the warehouse management module.
 */
class Shopware_Controllers_Backend_ViisonPickwareERPWarehouseManagement extends ViisonCommonBaseController
{
    /**
     * Responds a paginated, filtered and sorted list of warehouses.
     */
    public function getWarehouseListAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Update pefixes of sort fields
        foreach ($sort as &$sortField) {
            if (mb_strpos($sortField['property'], 'warehouse.') === false) {
                $sortField['property'] = 'warehouse.' . $sortField['property'];
            }
        }

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'warehouse'
        )->from(Warehouse::class, 'warehouse')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query', []);
        if (!empty($searchQuery)) {
            $builder->andWhere($builder->expr()->orX(
                'warehouse.code LIKE :searchQuery',
                'warehouse.name LIKE :searchQuery',
                'warehouse.contact LIKE :searchQuery',
                'warehouse.email LIKE :searchQuery'
            ));
            $builder->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
        ]);
    }

    /**
     * Creates one or more warehouses using the POSTed data.
     */
    public function createWarehousesAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // fix any previous problems with the default warehouse
        $this->get('pickware.erp.warehouse_integrity_service')->ensureSingleDefaultWarehouse();

        // Create a warehouse for each given data array
        $newWarehouses = [];
        foreach ($data as $warehouseData) {
            // Validate data
            if (!$this->validateWarehouseUniqueConstraints($warehouseData['name'], $warehouseData['code'])) {
                $this->View()->success = false;
                $this->View()->uniqueConstraintViolation = true;

                return;
            }

            // If the default flags are set, ensure that only one warehouse has one of the default flags enabled
            if ($warehouseData['defaultWarehouse']) {
                /** @var Warehouse[] $defaultWarehouses */
                $defaultWarehouses = $this->get('models')->getRepository(Warehouse::class)->findBy([
                    'defaultWarehouse' => true,
                ]);
                foreach ($defaultWarehouses as $warehouse) {
                    $warehouse->setDefaultWarehouse(false);
                }
            }
            if ($warehouseData['defaultReturnShipmentWarehouse']) {
                /** @var Warehouse[] $defaultReturnShipmentWarehouses */
                $defaultReturnShipmentWarehouses = $this->get('models')->getRepository(Warehouse::class)->findBy([
                    'defaultReturnShipmentWarehouse' => true,
                ]);
                foreach ($defaultReturnShipmentWarehouses as $warehouse) {
                    $warehouse->setDefaultReturnShipmentWarehouse(false);
                }
            }

            // Convert some fields
            if (isset($warehouseData['binLocations']) && is_array($warehouseData['binLocations'])) {
                $warehouseData['binLocations'] = new ArrayCollection($warehouseData['binLocations']);
            }

            // Create warehouse
            $warehouse = new Warehouse();
            $warehouse->fromArray($warehouseData);
            $this->get('models')->persist($warehouse);
            $newWarehouses[] = $warehouse;

            // Create and add a new default bin location
            $nullBinLocation = new BinLocation($warehouse, BinLocation::NULL_BIN_LOCATION_CODE);
            $warehouse->setNullBinLocation($nullBinLocation);
            $this->get('models')->persist($nullBinLocation);
        }

        // Save changes
        $this->get('models')->flush();

        // Assign all existing article details to the default bin locations of all new warehouses
        foreach ($newWarehouses as $warehouse) {
            $this->get('pickware.erp.warehouse_integrity_service')->ensureMappingsForAllArticleDetailsInWarehouse(
                $warehouse
            );
        }

        // Set a filter parameter using the IDs of the newly created warehouses and add the data of the
        // created warehouses to the response
        $this->Request()->setParam('filter', [
            [
                'property' => 'warehouse.id',
                'expression' => 'IN',
                'value' => array_map(
                    function ($warehouse) {
                        return $warehouse->getId();
                    },
                    $newWarehouses
                ),
            ],
        ]);
        $this->getWarehouseListAction();
    }

    /**
     * Updates one or more existing warehouses using the POSTed data.
     */
    public function updateWarehousesAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var WarehouseIntegrity $warehouseIntegrityService */
        $warehouseIntegrityService = $this->get('pickware.erp.warehouse_integrity_service');

        // fix any previous problems with the default warehouse
        $warehouseIntegrityService->ensureSingleDefaultWarehouse();
        $warehouseIntegrityService->ensureSingleDefaultReturnShipmentWarehouse();

        // Update all given warehouses with the respective data
        foreach ($data as $warehouseData) {
            unset($warehouseData['binLocations']);
            // Validate data
            if (!$this->validateWarehouseUniqueConstraints($warehouseData['name'], $warehouseData['code'], $warehouseData['id'])) {
                $this->View()->success = false;
                $this->View()->uniqueConstraintViolation = true;

                return;
            }
            /** @var Warehouse $warehouse */
            $warehouse = $entityManager->find(Warehouse::class, $warehouseData['id']);
            ParameterValidator::assertEntityFound($warehouse, Warehouse::class, $warehouseData['id'], 'id');

            // If the default flags are set, ensure that only one warehouse has one of the default flags enabled
            if ($warehouseData['defaultWarehouse']) {
                /** @var Warehouse[] $defaultWarehouses */
                $defaultWarehouses = $entityManager->getRepository(Warehouse::class)->findBy([
                    'defaultWarehouse' => true,
                ]);
                foreach ($defaultWarehouses as $defaultWarehouse) {
                    $defaultWarehouse->setDefaultWarehouse(false);
                }
                $entityManager->flush($defaultWarehouses);
                $warehouse->setDefaultWarehouse(true);
            }
            if ($warehouseData['defaultReturnShipmentWarehouse']) {
                /** @var Warehouse[] $defaultReturnShipmentWarehouses */
                $defaultReturnShipmentWarehouses = $entityManager->getRepository(Warehouse::class)->findBy([
                    'defaultReturnShipmentWarehouse' => true,
                ]);
                foreach ($defaultReturnShipmentWarehouses as $defaultReturnShipmentWarehouse) {
                    $defaultReturnShipmentWarehouse->setDefaultReturnShipmentWarehouse(false);
                }
                $entityManager->flush($defaultReturnShipmentWarehouses);
                $warehouse->setDefaultReturnShipmentWarehouse(true);
            }

            unset($warehouseData['defaultWarehouse']);
            unset($warehouseData['defaultReturnShipmentWarehouse']);

            $stockAvailabilityChanged = $warehouse->isStockAvailableForSale() != $warehouseData['stockAvailableForSale'];
            $warehouse->fromArray($warehouseData);
            $entityManager->flush($warehouse);
            if ($stockAvailabilityChanged) {
                $this->updateArticleDetailInStockForWarehouse($warehouse);
            }
        }

        // The above code may still result in a situation with no default warehouses (due to stale UI state). Fix this
        // by picking a new default warehouse.
        $warehouseIntegrityService->ensureSingleDefaultWarehouse();
        $warehouseIntegrityService->ensureSingleDefaultReturnShipmentWarehouse();

        // Update the list of warehouses
        $this->getWarehouseListAction();
    }

    /**
     * Deletes the POSTed warehouses.
     */
    public function deleteWarehousesAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Validate all warehouses whether they can be removed
        $warehouseConstraintViolations = [];
        $warehouseIntegrity = $this->get('pickware.erp.warehouse_integrity_service');
        foreach ($data as $warehouseData) {
            $warehouse = $this->get('models')->find(Warehouse::class, $warehouseData['id']);
            if (!$warehouse) {
                continue;
            }
            $constraintViolations = $warehouseIntegrity->evaluateWarehouseDeletionConstraints($warehouse);
            if (count($constraintViolations) > 0) {
                $warehouseConstraintViolations[$warehouse->getId()] = $constraintViolations;
            }
        }
        if (count($warehouseConstraintViolations) > 0) {
            // At least one warehouse cannot be removed, hence respond with the constraint violation
            $this->View()->assign([
                'success' => false,
                'issues' => $warehouseConstraintViolations,
            ]);

            return;
        }

        // Remove all given warehouses
        $defaultWarehouse = $this->get('models')->getRepository(Warehouse::class)->getDefaultWarehouse();
        foreach ($data as $warehouseData) {
            $warehouse = $this->get('models')->find(Warehouse::class, $warehouseData['id']);
            if (!$warehouse) {
                continue;
            }

            // Find all supplier orders that were shipped to the warehouse and change their
            // warehouse to the default warehouse
            $this->get('db')->query(
                'UPDATE `pickware_erp_supplier_orders`
                SET warehouseId = :defaultWarehouseId
                WHERE warehouseId = :warehouseId',
                [
                    'defaultWarehouseId' => $defaultWarehouse->getId(),
                    'warehouseId' => $warehouse->getId(),
                ]
            );

            // Remove the warehouse and all its associated entities
            $this->get('models')->remove($warehouse);
            $this->get('models')->flush($warehouse);
        }

        $this->View()->success = true;
    }

    /**
     * Responds a paginated, filtered and sorted list of bin locations.
     */
    public function getBinLocationListAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Update pefixes of sort fields
        foreach ($sort as $index => &$sortField) {
            if ($sortField['property'] !== 'isNullBinLocation' && mb_strpos($sortField['property'], '.') === false) {
                $sortField['property'] = 'binLocation.' . $sortField['property'];
            }
        }

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder
            ->select(
                'binLocation',
                'CASE WHEN binLocation.id = warehouse.nullBinLocationId THEN 1 ELSE 0 END AS HIDDEN isNullBinLocation'
            )
            ->from(BinLocation::class, 'binLocation')
            ->join('binLocation.warehouse', 'warehouse')
            ->leftJoin('binLocation.articleDetailBinLocationMappings', 'articleDetailBinLocationMappings')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Check for a search query
        $searchQuery = $this->Request()->getParam('query');
        if (!empty($searchQuery)) {
            $builder->andWhere('binLocation.code LIKE :searchQuery')
                    ->setParameter('searchQuery', ('%' . $searchQuery . '%'));
        }

        // Check whether to include the warehouse data for each bin location
        $includeWarehouseData = !!$this->Request()->getParam('includeWarehouseData', false);
        if ($includeWarehouseData) {
            $builder->addSelect('warehouse');
        }

        // Check whether to include the default bin locations in the results
        $includeNullLocations = !!$this->Request()->getParam('includeNullLocations', false);
        if (!$includeNullLocations) {
            $builder->andWhere('binLocation.id != warehouse.nullBinLocationId');
        }

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $totalResult = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
        ]);
    }

    /**
     * Creates one or more bin locations using the POSTed data.
     */
    public function createBinLocationsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Create a bin location for each given data array
        $newBinLocations = [];
        foreach ($data as $binLocationData) {
            // Validate data
            if (!$this->validateBinLocationUniqueConstraints($binLocationData['code'], $binLocationData['warehouseId'])) {
                $this->View()->success = false;
                $this->View()->uniqueConstraintViolation = true;

                return;
            }

            // Create binLocation
            $warehouse = $this->get('models')->find(Warehouse::class, $binLocationData['warehouseId']);
            $binLocation = new BinLocation($warehouse, $binLocationData['code']);
            $binLocation->fromArray($binLocationData);
            $this->get('models')->persist($binLocation);
            $newBinLocations[] = $binLocation;
        }

        // Save changes
        $this->get('models')->flush();

        // Set a filter parameter using the IDs of the newly created bin locations and add the data of the
        // created bin locations to the response
        $this->Request()->setParam('filter', [
            [
                'property' => 'binLocation.id',
                'expression' => 'IN',
                'value' => array_map(
                    function ($binLocation) {
                        return $binLocation->getId();
                    },
                    $newBinLocations
                ),
            ],
        ]);
        $this->getBinLocationListAction();
    }

    /**
     * Updates one or more existing bin locations using the POSTed data.
     */
    public function updateBinLocationsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Create a bin location for each given data array
        $newBinLocations = [];
        foreach ($data as $binLocationData) {
            // Validate data
            if (!$this->validateBinLocationUniqueConstraints($binLocationData['code'], $binLocationData['warehouseId'], $binLocationData['id'])) {
                $this->View()->success = false;
                $this->View()->uniqueConstraintViolation = true;

                return;
            }

            // Prepare the data
            if (isset($binLocationData['warehouseId'])) {
                $binLocationData['warehouse'] = $this->get('models')->find(Warehouse::class, $binLocationData['warehouseId']);
                unset($binLocationData['warehouseId']);
            }

            // Update bin location
            $binLocation = $this->get('models')->find(BinLocation::class, $binLocationData['id']);
            if ($binLocation) {
                $binLocation->fromArray($binLocationData);
            }
        }

        // Save changes
        $this->get('models')->flush();

        // Set a filter parameter using the IDs of the updated bin locations and add the data of the
        // updated bin locations to the response
        $this->Request()->setParam('filter', [
            [
                'property' => 'binLocation.id',
                'expression' => 'IN',
                'value' => array_map(
                    function ($binLocationData) {
                        return $binLocationData['id'];
                    },
                    $data
                ),
            ],
        ]);
        $this->getBinLocationListAction();
    }

    /**
     * Deletes the POSTed bin locations.
     */
    public function deleteBinLocationsAction()
    {
        // Load data
        $data = $this->Request()->getParam('data');
        if (!$data) {
            $this->View()->success = false;

            return;
        }

        // Remove all given bin locations
        foreach ($data as $binLocationsData) {
            $binLocation = $this->get('models')->find(BinLocation::class, $binLocationsData['id']);
            if ($binLocation && $binLocation->getCode() !== BinLocation::NULL_BIN_LOCATION_CODE) {
                $this->deleteBinLocation($binLocation);
            }
        }

        // Save changes
        $this->get('models')->flush();

        $this->View()->success = true;
    }

    /**
     * Responds a paginated, filtered and sorted list of bin location article detail mapping data.
     */
    public function getBinLocationArticleDetailListAction()
    {
        $limit = $this->Request()->getParam('limit', 1000);
        $offset = $this->Request()->getParam('start', 0);
        $sort = $this->Request()->getParam('sort', []);
        $filter = $this->Request()->getParam('filter', []);

        // Map sort fields
        $mapping = [
            'name' => 'article.name',
            'number' => 'articleDetail.number',
            'stock' => 'binLocationMapping.stock',
        ];
        foreach ($sort as &$sortField) {
            if (isset($mapping[$sortField['property']])) {
                $sortField['property'] = $mapping[$sortField['property']];
            }
        }

        // Build the main query
        $builder = $this->get('models')->createQueryBuilder();
        $builder->select(
            'article.id AS articleId',
            'article.name AS name',
            'articleDetail.id AS articleDetailId',
            'articleDetail.number AS number',
            'binLocationMapping.stock AS stock'
        )->from(ArticleDetailBinLocationMapping::class, 'binLocationMapping')
            ->join('binLocationMapping.articleDetail', 'articleDetail')
            ->join('articleDetail.article', 'article')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->setFirstResult($offset)
            ->setMaxResults($limit);

        // Create the query and execute it to get the paginated results
        $query = $builder->getQuery();
        $query->setHydrationMode(Query::HYDRATE_ARRAY);
        $paginator = new Paginator($query);
        $paginator->setUseOutputWalkers(false);
        $totalResult = $paginator->count();
        $result = $paginator->getIterator()->getArrayCopy();

        // Add additional text to each result's name
        $additionalTexts = ViisonCommonUtil::getVariantAdditionalTexts(array_column($result, 'articleDetailId'));
        foreach ($result as &$binLocationMapping) {
            $additionalText = $additionalTexts[$binLocationMapping['articleDetailId']];
            $binLocationMapping['name'] .= $additionalText ? ' - ' . $additionalText : '';
        }

        $this->View()->assign([
            'success' => true,
            'data' => $result,
            'total' => $totalResult,
        ]);
    }

    /**
     * Validates the POSTed components and uses them to create all possible, not yet existing
     * bin locations, which are then added to the warehouse with the given ID.
     */
    public function generateBinLocationsAction()
    {
        // Get and check warehouse ID and component data
        $warehouseId = $this->Request()->getParam('warehouseId');
        $components = $this->Request()->getParam('components');
        if (empty($warehouseId) || empty($components) || !is_array($components)) {
            $this->View()->success = false;

            return;
        }

        // Try to find the warehouse
        $warehouse = $this->get('models')->find(Warehouse::class, $warehouseId);
        if (!$warehouse) {
            $this->View()->success = false;
            $this->View()->message = 'Warehouse with ID ' . $warehouseId . ' not found.';

            return;
        }

        // Get all currently existing bin locations of the given warehouse
        $existingBinLocations = $this->get('db')->fetchCol(
            'SELECT code
            FROM `pickware_erp_bin_locations`
            WHERE warehouseId = ?',
            [
                $warehouseId
            ]
        );

        // Create all bin locations (in batches) and add them to the warehouse
        // if they don't exist yet
        try {
            $binLocationCodeComponent = $this->get('pickware.erp.bin_location_code_generator_service')->createLinkedCodeComponent(
                $components
            );
        } catch (BinLocationCodeGeneratorException $exception) {
            $this->View()->success = false;
            $this->View()->message = 'Invalid bin location code format.';

            return;
        }
        $batchSize = 1000;
        $numGeneratedBinLocations = 0;
        $numSkippedBinLocations = 0;
        do {
            // Generate the next batch of bin locations and remove all duplicates
            $binLocationsBatch = $binLocationCodeComponent->createCodes($batchSize);
            $originalCount = count($binLocationsBatch);
            $binLocationsBatch = array_diff($binLocationsBatch, $existingBinLocations);
            $numGeneratedBinLocations += count($binLocationsBatch);
            $numSkippedBinLocations += $originalCount - count($binLocationsBatch);
            if (empty($binLocationsBatch)) {
                continue;
            }

            // Generate a query INSERTing the whole batch into the database at once,
            // since this is ~10x faster than 'batch' processing using Doctrine. For some
            // reason (maybe the unique constraint) Doctrine performs a single INSERT
            // query for each record even when calling 'flush()' only once per batch.
            $sql = 'INSERT INTO `pickware_erp_bin_locations` (warehouseId, code) VALUES ';
            $sql .= implode(
                ', ',
                array_map(
                    function ($binLocationCode) use ($warehouseId) {
                        $binLocationCode = $this->get('db')->quote($binLocationCode);

                        return ' (' . intval($warehouseId) . ', ' . $binLocationCode . ')';
                    },
                    $binLocationsBatch
                )
            );
            $this->get('db')->query($sql);
        } while ($binLocationCodeComponent->hasMoreCodes());

        $this->View()->assign([
            'success' => true,
            'numGeneratedBinLocations' => $numGeneratedBinLocations,
            'numSkippedBinLocations' => $numSkippedBinLocations,
        ]);
    }

    /**
     * Validates the POSTed components and saves them in the warehouse with the given ID.
     */
    public function saveBinLocationFormatComponentsAction()
    {
        // Try to find the warehouse
        $warehouseId = $this->Request()->getParam('warehouseId', 0);
        $warehouse = $this->get('models')->find(Warehouse::class, $warehouseId);
        if (!$warehouse) {
            $this->View()->success = false;
            $this->View()->message = 'Warehouse with ID ' . $warehouseId . ' not found.';

            return;
        }

        // Get and validate the components
        $components = $this->Request()->getParam('components');
        $components = (is_array($components) && count($components) > 0) ? $components : null;
        if ($components && !$this->get('pickware.erp.bin_location_code_generator_service')->areRawCodeComponentsValid($components)) {
            $this->View()->success = false;
            $this->View()->message = 'Invalid bin location format.';

            return;
        }

        // Save the format components
        $warehouse->setBinLocationFormatComponents($components);
        $this->get('models')->flush($warehouse);

        $this->View()->success = true;
    }

    /**
     * Uses the POSTed components to create a single bin location code as well as to calculate the number of bin
     * location codes possible for the code components.
     */
    public function getExampleBinLocationAction()
    {
        // Get and check component data
        $components = $this->Request()->getParam('components');
        if (empty($components) || !is_array($components)) {
            $this->View()->success = false;

            return;
        }

        try {
            // Create a single bin location and determine the number of possible bin locations
            $binLocationCodeComponent = $this->get('pickware.erp.bin_location_code_generator_service')->createLinkedCodeComponent(
                $components
            );
            $binLocations = $binLocationCodeComponent->createCodes(1);
            $numPossibleLocations = $binLocationCodeComponent->getNumberOfPossibleCodes();

            $this->View()->assign([
                'success' => true,
                'binLocation' => $binLocations[0],
                'numPossibleLocations' => $numPossibleLocations,
            ]);
        } catch (BinLocationCodeGeneratorException $exception) {
            $this->View()->success = false;
            $this->View()->message = 'Invalid bin location code format.';
        }
    }

    /**
     * Deletes the given bin location from the database, after performing all necessary steps
     * to ensure the integrity of bin location article detail mappings and stock snapshots.
     * That is, first a relocation from the deleted bin location to default bin location is logged,
     * to move all the stock away from the location. Then all stock entries, whose snapshots contain
     * the bin location that is about to be deleted, are loaded and their snapshots are cleaned
     * from the given bin location. As a result, each of these snapshots contains the default
     * bin location afterwards and the stock af that default bin location snapshot is increased
     * by the stock of the deleted bin location snapshot.
     *
     * Note: This method does flush the created/updated/removed entities, so you must call 'flush()'
     *       after calling this method.
     *
     * @param BinLocation $binLocation
     */
    private function deleteBinLocation(BinLocation $binLocation)
    {
        // Relocate all articles, which are mapped to the bin location, to the default location
        $relocationComment = 'Löschen von Lagerplatz "' . $binLocation->getCode() . '"';
        $nullBinLocation = $binLocation->getWarehouse()->getNullBinLocation();
        foreach ($binLocation->getArticleDetailBinLocationMappings() as $mapping) {
            $stockChanges = new RelocationStockChangeList([
                new BinLocationStockChange($binLocation, -1 * $mapping->getStock())
            ], new BinLocationStockChange($nullBinLocation, $mapping->getStock()));
            $this->get('pickware.erp.stock_ledger_service')->recordRelocatedStock(
                $mapping->getArticleDetail(),
                $stockChanges,
                $relocationComment
            );
        }

        // Determine all stock entries, whose snapshots must be adopted
        $stockEntries = $binLocation->getStockSnapshots()->map(function ($snapshot) {
            return $snapshot->getStockLedgerEntry();
        });
        foreach ($stockEntries as $stockEntry) {
            // Try to find the snapshots of the deleted bin location and the default bin location
            $deletedLocationSnapshot = null;
            $defaultLocationSnapshot = null;
            foreach ($stockEntry->getBinLocationStockSnapshots() as $snapshot) {
                if ($snapshot->getBinLocation()->getId() === $binLocation->getId()) {
                    $deletedLocationSnapshot = $snapshot;
                } elseif ($snapshot->getBinLocation()->getId() === $nullBinLocation->getId()) {
                    $defaultLocationSnapshot = $snapshot;
                }
            }
            if (!$defaultLocationSnapshot) {
                // Default bin location was not a part of the snapshot, hence create a new snapshot
                $defaultLocationSnapshot = new BinLocationStockSnapshot(
                    $deletedLocationSnapshot->getStockLedgerEntry(),
                    $nullBinLocation,
                    0
                );
                $this->get('models')->persist($defaultLocationSnapshot);
            }

            // Add the stock of the deleted bin location snapshot to the default location snapshot
            $defaultLocationSnapshot->setStock($defaultLocationSnapshot->getStock() + $deletedLocationSnapshot->getStock());

            // Remove the snapshot of the deleted bin location
            $this->get('models')->remove($deletedLocationSnapshot);
        }

        // Remove the bin location
        $this->get('models')->remove($binLocation);
    }

    /**
     * Checks whether the stock of the given $warehouse should be available and, based on
     * that information, either increases or decreases both 'inStock' and 'viisonPhyisicalStockForSale'
     * of all article details by the respective stock in the given warehouse.
     *
     * @param Warehouse $warehouse
     */
    private function updateArticleDetailInStockForWarehouse(Warehouse $warehouse)
    {
        $operator = ($warehouse->isStockAvailableForSale()) ? '+' : '-';
        $sql = sprintf(
            'UPDATE s_articles_details ad
            LEFT JOIN s_articles_attributes att
                ON att.articledetailsID = ad.id
            SET
                ad.instock = ad.instock %1$s (
                    SELECT `stockCounts`.stock
                    FROM `pickware_erp_warehouse_article_detail_stock_counts` AS `stockCounts`
                    WHERE `stockCounts`.articleDetailId = ad.id
                    AND `stockCounts`.warehouseId = :warehouseId
                ),
                att.pickware_physical_stock_for_sale = att.pickware_physical_stock_for_sale %1$s (
                    SELECT `stockCounts`.stock
                    FROM `pickware_erp_warehouse_article_detail_stock_counts` AS `stockCounts`
                    WHERE `stockCounts`.articleDetailId = ad.id
                    AND `stockCounts`.warehouseId = :warehouseId
                )
            ',
            $operator
        );
        $this->get('db')->query(
            $sql,
            [
                'warehouseId' => $warehouse->getId(),
            ]
        );
    }

    /**
     * Checks for any warehouses (other than the one with the given ID) having the
     * given name and/or code and validates whether such a warehouse exists.
     *
     * @param string $name
     * @param string $code
     * @param int $warehouseId (optional)
     * @return boolean
     */
    private function validateWarehouseUniqueConstraints($name, $code, $warehouseId = 0)
    {
        // Try to find warehouses having the same name or code as given
        $existingWarehouseIds = $this->get('db')->fetchCol(
            'SELECT id
            FROM `pickware_erp_warehouses`
            WHERE id != :warehouseId
            AND (
                name = :name
                OR code = :code
            )',
            [
                'warehouseId' => $warehouseId,
                'name' => $name,
                'code' => $code,
            ]
        );

        return empty($existingWarehouseIds);
    }

    /**
     * Checks for any bin location (other than the one with the given ID) having the
     * given code and warehouse ID and validates whether such a bin location exists.
     *
     * @param string $code
     * @param string $warehouseId
     * @param int $binLocationId (optional)
     * @return boolean
     */
    private function validateBinLocationUniqueConstraints($code, $warehouseId, $binLocationId = 0)
    {
        // Try to find bin locations having the same code and warehouse ID as given
        $existingBinLocationIds = $this->get('db')->fetchCol(
            'SELECT id
            FROM `pickware_erp_bin_locations`
            WHERE id != :binLocationId
            AND code = :code
            AND warehouseId = :warehouseId',
            [
                'binLocationId' => $binLocationId,
                'code' => $code,
                'warehouseId' => $warehouseId,
            ]
        );

        return empty($existingBinLocationIds);
    }
}
