<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Migrations;

use Shopware\Models\Mail\Mail;
use Shopware\Models\Plugin\Plugin;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\Migration\Migration;
use Shopware\Plugins\ViisonCommon\Components\Migration\MigrationTranscript;
use Shopware_Plugins_Core_ViisonPickwareERP_Bootstrap as ErpBootstrap;

/**
 * Find the mail templates for the ERP document mailer and migrates them to templates used by Shopware's built in
 * order document mailer.
 *
 * Migration is executable since Shopware 5.5.0.
 */
class DocumentMailTemplatesToBuiltInMailTemplatesMigration implements Migration
{
    /**
     * @var \Zend_Db_Adapter_Abstract
     */
    private $db;

    /**
     * @var Plugin
     */
    private $plugin;

    /**
     * @var \Enlight_Components_Snippet_Manager
     */
    private $bootstrapSnippetManager;

    /**
     * @param \Zend_Db_Adapter_Abstract $db
     * @param Plugin $plugin
     * @param \Enlight_Components_Snippet_Manager $bootstrapSnippetManager
     */
    public function __construct($db, Plugin $plugin, \Enlight_Components_Snippet_Manager $bootstrapSnippetManager)
    {
        $this->db = $db;
        $this->plugin = $plugin;
        $this->bootstrapSnippetManager = $bootstrapSnippetManager;
    }

    /**
     * @inheritdoc
     */
    public function canExecute()
    {
        return ViisonCommonUtil::assertMinimumShopwareVersion('5.5.0');
    }

    /**
     * @inheritdoc
     */
    public function execute(MigrationTranscript $transcript)
    {
        // On a new install do not overwrite the mail templates. We are finished with the migration here.
        if ($this->plugin->getInstalled() === null) {
            return;
        }

        $documentMailerMailTemplateNamePrefix = ErpBootstrap::DOCUMENT_MAILER_MAIL_TEMPLATE_NAME . '_DocType';
        $documentMailerMailTemplateRows = $this->db->fetchAll(
            'SELECT * FROM `s_core_config_mails` WHERE `name` LIKE :mailTemplateNamePrefix',
            [
                'mailTemplateNamePrefix' => $documentMailerMailTemplateNamePrefix . '%',
            ]
        );

        $documentNamesOfMigratedDocumentMailTemplates = [];
        foreach ($documentMailerMailTemplateRows as $documentMailerMailTemplateRow) {
            $documentTypeId = (int) str_replace($documentMailerMailTemplateNamePrefix, '', $documentMailerMailTemplateRow['name']);
            
            $documentTypeRow = $this->db->fetchRow(
                'SELECT * FROM `s_core_documents` WHERE `id` = :documentTypeId',
                [
                    'documentTypeId' => $documentTypeId,
                ]
            );

            if ($documentTypeRow === false || empty($documentTypeRow['key'])) {
                $transcript->getLogger()->warning(sprintf(
                    'Migration warning: Mail template with name "%s" could not be migrated to Shopware\'s built in ' .
                    'document mail templates format because the corresponding document type was not found or it has ' .
                    'no valid key.',
                    $documentMailerMailTemplateRow['name']
                ), [
                    'migrationClass' => get_class($this),
                    'documentTypeId' => $documentTypeId,
                ]);

                continue;
            }

            // Create a new mail template for this document type or overwrite the default one brought with Shopware.
            $newMailTemplateName = 'document_' . $documentTypeRow['key'];

            $existingMailTemplate = $this->db->fetchRow(
                'SELECT * FROM `s_core_config_mails` WHERE `name` = :newMailTemplateName',
                ['newMailTemplateName' => $newMailTemplateName]
            );

            // Backup an existing mail template for this document, so the user can restore it on any failure or unwanted migration.
            if ($existingMailTemplate !== false) {
                $backupHintSnippet = $this->bootstrapSnippetManager->getNamespace('viison_pickware_erp/migrations/document_mail_templates_to_built_in_mail_templates_migration')->get('mailTemplateBackupPrefix');
                $contentPrefix = implode("\n", [
                    '# ',
                    '# ' . sprintf($backupHintSnippet, $newMailTemplateName),
                    '# ',
                ]);

                $this->db->query(
                    'UPDATE `s_core_config_mails`
                SET
                    `name` = CONCAT(`name`, " (Backup) ", NOW()),
                    `mailtype` = 1,
                    `content` = CONCAT(:contentPrefix, `content`)
                WHERE `name` = :newMailTemplateName',
                    [
                        'newMailTemplateName' => $newMailTemplateName,
                        'contentPrefix' => $contentPrefix . "\n",
                    ]
                );
            }

            $this->db->query(
                'UPDATE 
                    `s_core_config_mails`
                SET 
                    mailType = :mailType,
                    name = :newMailTemplateName,
                    frommail = :frommail,
                    fromname = :fromname,
                    subject = :subject,
                    content = :content,
                    contentHTML = :contentHTML,
                    context = null
                WHERE 
                    id = :id',
                [
                    'id' => $documentMailerMailTemplateRow['id'],
                    'mailType' => Mail::MAILTYPE_DOCUMENT,
                    'newMailTemplateName' => $newMailTemplateName,
                    'frommail' => self::migrateTemplateVariables($documentMailerMailTemplateRow['frommail']),
                    'fromname' => self::migrateTemplateVariables($documentMailerMailTemplateRow['fromname']),
                    'subject' => self::migrateTemplateVariables($documentMailerMailTemplateRow['subject']),
                    'content' => self::migrateTemplateVariables($documentMailerMailTemplateRow['content']),
                    'contentHTML' => self::migrateTemplateVariables($documentMailerMailTemplateRow['contentHTML']),
                ]
            );

            // Migrate the translations of the mail template.
            $translations = $this->db->fetchAll(
                'SELECT id, objectdata
                FROM `s_core_translations`
                WHERE `objectkey` = :mailTemplateId
                AND `objecttype` = "config_mails"',
                [
                    'mailTemplateId' => $documentMailerMailTemplateRow['id'],
                ]
            );

            foreach ($translations as $oldTranslation) {
                // Migrate the template variable names, as they differ.
                $oldObjectData = unserialize($oldTranslation['objectdata']);
                $newObjectData = array_map(function ($objectElement) {
                    return self::migrateTemplateVariables($objectElement);
                }, $oldObjectData);
                $newObjectDataSerialized = serialize($newObjectData);

                $this->db->query(
                    'UPDATE 
                        `s_core_translations`
                    SET 
                        objectdata = :objectData
                    WHERE
                        id = :id',
                    [
                        'objectData' => $newObjectDataSerialized,
                        'id' => $oldTranslation['id'],
                    ]
                );
            }

            $documentNamesOfMigratedDocumentMailTemplates[] = $documentTypeRow['name'];
        }

        if (count($documentNamesOfMigratedDocumentMailTemplates) !== 0) {
            $transcript->addLocalizedMessage(
                'migrationHint',
                'viison_pickware_erp/migrations/document_mail_templates_to_built_in_mail_templates_migration',
                [
                    implode(', ', $documentNamesOfMigratedDocumentMailTemplates),
                ]
            );
        }
    }

    /**
     * Replaces all template variables in a template of the ERP order document mailer with the ones used in the Shopware
     * document mailer.
     *
     * @param string $templateString
     * @return string
     */
    private static function migrateTemplateVariables($templateString)
    {
        $replacePairs = [
            '$orderNumber' => '$sOrder.ordernumber',
            '$userSalutation' => '$sUser.salutation',
            '$userFirstName' => '$sUser.firstname',
            '$userLastName' => '$sUser.lastname',
        ];

        return strtr($templateString, $replacePairs);
    }
}
