<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\Analytics;

use DateTime;
use Shopware\Components\Model\DBAL\Result as DBALResult;
use Shopware\Models\Analytics\Repository as AnalyticsRepository;
use Shopware\Models\Article\Article;
use Shopware\Models\Category\Category;
use Shopware\Models\Shop\Shop;

class Repository extends AnalyticsRepository
{
    /**
     * Returns the sales of articles grouped by main shop categories. These are the first child categories of the active
     * default shop.
     *
     * Articles may have multiple categories assigned to them. We first map articles down to the respective main shop
     * categories, then select the category with the lowest position ("first"/"most important" main shop category).
     *
     * @param DateTime|null $from
     * @param DateTime|null $to
     * @return DBALResult
     */
    public function getArticleSalesPerMainShopCategory(DateTime $from = null, DateTime $to = null)
    {
        $entityManager = Shopware()->Container()->get('models');
        $databaseConnection = $this->connection;

        /* @var $defaultShop Shop */
        $defaultShop = $entityManager->getRepository(Shop::class)->getDefault();
        $rootCategoryId = $defaultShop->getCategory()->getId();

        // Do correctly map any category down to a main shop category, we need to build up the full category tree
        // beforehand. Since it is not possible via SQL, do it programmatically.
        $builder = $entityManager->createQueryBuilder();
        $builder
            ->select(
                'category.id',
                'category.parentId',
                'category.position'
            )
            ->from(Category::class, 'category');
        $categories = $builder->getQuery()->getArrayResult();

        // Add main shop category id to the category result to create a (category -> main shop category) mapping
        $mainShopCategoryMapping = $this->addMainShopCategoryMapping($categories, $rootCategoryId);

        // Fetch all (article -> category) mappings
        $builder = $entityManager->createQueryBuilder();
        $builder
            ->select(
                'article.id as articleId',
                'category.id as categoryId'
            )
            ->from(Article::class, 'article')
            ->leftJoin('article.categories', 'category');
        $articleCategoryMappings = $builder->getQuery()->getArrayResult();

        // Improve (category -> main shop category) mapping to (article -> main shop category) mapping with the lowest
        // main shop category position if any exist.
        $articleMainShopCategoryMappings = [];
        foreach ($articleCategoryMappings as $articleCategoryMapping) {
            $articleId = $articleCategoryMapping['articleId'];
            $categoryId = $articleCategoryMapping['categoryId'];
            $mainShopCategory = $mainShopCategoryMapping[$categoryId];

            // Set the main shop category if it is the first one found or has a lower position than existing one.
            $useMainShopCategory = (
                !isset($articleMainShopCategoryMappings[$articleId]['mainShopCategoryId'])
                || (
                    isset($mainShopCategory['mainShopCategoryPosition']) &&
                    $mainShopCategory['mainShopCategoryPosition'] < $articleMainShopCategoryMappings[$articleId]['mainShopCategoryPosition']
                )
            );
            if ($useMainShopCategory) {
                $articleMainShopCategoryMappings[$articleId] = $mainShopCategory;
            }
            $articleMainShopCategoryMappings[$articleId]['articleId'] = $articleId; // Add article id key as field
        }

        // Create temporary table with the (article -> main shop category) mapping information
        $insertionValueStrings = array_map(
            function ($mapping) {
                return '(' . $mapping['articleId'] . ', ' . ($mapping['mainShopCategoryId'] ?: 'NULL') . ')';
            },
            $articleMainShopCategoryMappings
        );
        $categoryMappingTableValueString = implode(', ', $insertionValueStrings);
        $databaseConnection->query(
            'CREATE TEMPORARY TABLE s_viison_pickware_erp_tmp_category_mapping (
                articleId int,
                mainShopCategoryId int,
                INDEX (`articleId`)
            )'
        );
        $databaseConnection->query('INSERT INTO s_viison_pickware_erp_tmp_category_mapping (articleId, mainShopCategoryId) VALUES ' . $categoryMappingTableValueString);

        // Build the actual querybuilder that groups sales by main shop category
        $builder = $this->createProductAmountBuilder($from, $to)
            ->addSelect('mainShopCategory.description as name')
            ->addSelect('mainShopCategory.id as node')
            ->innerJoin('articles', 's_viison_pickware_erp_tmp_category_mapping', 'mainShopCategoryMappings', 'articles.id = mainShopCategoryMappings.articleId')
            ->innerJoin('mainShopCategoryMappings', 's_categories', 'mainShopCategory', 'mainShopCategoryMappings.mainShopCategoryId = mainShopCategory.id')
            ->andWhere('mainShopCategory.active = 1')
            ->groupBy('mainShopCategory.id');

        $select = $builder->getQueryPart('select');
        $select[1] = 'SUM((details.price*(1+IF(orders.net, IF(orders.taxfree, 0, details.tax_rate), 0)/100) * details.quantity)/currencyFactor) AS turnover';
        $builder->select($select);

        $builder = $this->eventManager->filter('Shopware_Plugins_ViisonPickwareERP_ViisonPickwareERPAnalyticsMainShopCategory_ProductAmountPerMainShopCategory', $builder, [
            'subject' => $this,
        ]);

        return new DBALResult($builder);
    }

    /**
     * Adds the respective main shop category to the given array of categories, if the category is part of the
     * category subtree of the given $rootCategoryId. (Otherwise the main shop category is set to null)
     *
     * @param array $categories
     * @param int $rootCategoryId
     * @return array
     */
    private function addMainShopCategoryMapping(array $categories, $rootCategoryId)
    {
        // Map categories by their own id
        $categoryIds = array_map(function ($category) {
            return $category['id'];
        }, $categories);
        $categories = array_combine($categoryIds, $categories);

        $mainShopCategories = [];
        foreach ($categories as $categoryId => &$category) {
            $mainShopCategory = $this->getMainShopCategory($categoryId, $categories, $mainShopCategories, $rootCategoryId);
            $category['mainShopCategoryId'] = $mainShopCategory ? $mainShopCategory['id'] : null;
            $category['mainShopCategoryPosition'] = $mainShopCategory ? $mainShopCategory['position'] : null;

            // Add mapping result to array to minimize recursive calls
            if ($mainShopCategory) {
                $mainShopCategories[$categoryId] = $mainShopCategory;
            }
        }

        return $categories;
    }

    /**
     * Returns the main shop category that the given category is part of.
     *
     * @param int $categoryId
     * @param array $categories
     * @param array $mainShopCategories
     * @param int $rootCategoryId
     * @return int|null
     */
    private function getMainShopCategory($categoryId, array $categories, array $mainShopCategories, $rootCategoryId)
    {
        $category = $categories[$categoryId];

        if (!$category) {
            // Category was not found (should not happen)
            return null;
        } elseif ($category['parentId'] === null) {
            // Parent is missing. We skipped a main shop category (should not happen)
            return null;
        } elseif ($category['parentId'] === $rootCategoryId) {
            // Parent is root category. This is a main shop category
            return $categories[$categoryId];
        } elseif (array_key_exists($category['parentId'], $mainShopCategories)) {
            // Parent was already assigned to a main shop category. Use shortcut
            return $mainShopCategories[$category['parentId']];
        } else {
            // Step up recursively to the parent
            return $this->getMainShopCategory($category['parentId'], $categories, $mainShopCategories, $rootCategoryId);
        }
    }
}
