<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel;

use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabel;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\GroupableBarcodeLabel;

/**
 * A barcode label that identifies an article (variant), marked for printing.
 *
 * @ORM\Entity(repositoryClass="Repository")
 * @ORM\Table(name="pickware_erp_article_detail_barcode_labels")
 */
class ArticleBarcodeLabel extends ModelEntity implements BarcodeLabel, GroupableBarcodeLabel
{
    /**
     * @var int $id
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var int $articleDetailId
     *
     * @ORM\Column(name="articleDetailId", type="integer", options={"unsigned"=true}, nullable=false)
     */
    private $articleDetailId;

    /**
     * OWNING SIDE
     *
     * @var ArticleDetail $articleDetail the article variant to print a barcode label for
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Article\Detail")
     * @ORM\JoinColumn(name="articleDetailId", referencedColumnName="id", nullable=false)
     */
    protected $articleDetail;

    /**
     * @var int $quantity how many labels to print for this article
     *
     * @ORM\Column(name="quantity", type="integer", nullable=false)
     */
    private $quantity;

    /**
     * @var \DateTime $added
     *
     * @ORM\Column(name="added", type="datetime", nullable=false)
     */
    private $added;

    /**
     * BarcodeLabel constructor.
     *
     * @param ArticleDetail $articleDetail the article detail to print a barcode label for
     * @param int $quantity the number of barcode labels to print
     */
    public function __construct(ArticleDetail $articleDetail, $quantity = 1)
    {
        $this->articleDetail = $articleDetail;
        $this->articleDetailId = $articleDetail->getId();
        $this->quantity = $quantity;
        $this->added = new \DateTime();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return ArticleDetail the article to print a barcode label for
     */
    public function getArticleDetail()
    {
        return $this->articleDetail;
    }

    /**
     * @param ArticleDetail $articleDetail the article to print a barcode label for
     */
    public function setArticleDetail(ArticleDetail $articleDetail)
    {
        $this->articleDetail = $articleDetail;
        $this->articleDetailId = $articleDetail->getId();
    }

    /**
     * @return int
     */
    public function getArticleDetailId()
    {
        return $this->articleDetailId;
    }

    /**
     * @return int how many barcode labels to print
     */
    public function getQuantity()
    {
        return $this->quantity;
    }

    /**
     * @param int $quantity how many barcode labels to print
     */
    public function setQuantity($quantity)
    {
        $this->quantity = $quantity;
    }

    /**
     * Adds the given quantity to the quantity of this label.
     *
     * @param int $quantity how many more barcode labels to print
     */
    public function addQuantity($quantity)
    {
        $this->quantity += $quantity;
    }

    /**
     * @return \DateTime when this barcode label was marked for printing
     */
    public function getAdded()
    {
        return $this->added;
    }

    /**
     * @param \DateTime $added when this barcode label was marked for printing
     */
    public function setAdded(\DateTime $added)
    {
        $this->added = $added;
    }

    /**
     * @inheritdoc
     */
    public function getIdentifier()
    {
        return $this->getArticleDetailId();
    }
}
