<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use InvalidArgumentException;
use Shopware\Components\Model\ModelEntity;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabelGridPaperLayout;

/**
 * @ORM\Entity(repositoryClass="Repository")
 * @ORM\Table(name="pickware_erp_barcode_label_presets")
 */
class BarcodeLabelPreset extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="type", type="string", nullable=false)
     */
    private $type;

    /**
     * @var string
     *
     * @ORM\Column(name="name", type="string", nullable=false)
     */
    private $name;

    /**
     * @var string
     *
     * @ORM\Column(name="paperLayoutIdentifier", type="string", nullable=false)
     */
    private $paperLayoutIdentifier;

    /**
     * @var string
     *
     * @ORM\Column(name="templateIdentifier", type="string", nullable=false)
     */
    private $templateIdentifier;

    /**
     * @var float
     *
     * @ORM\Column(name="paperHeight", type="float", nullable=true)
     */
    private $paperHeight = null;

    /**
     * @var float
     *
     * @ORM\Column(name="paperWidth", type="float", nullable=true)
     */
    private $paperWidth = null;

    /**
     * @var float
     *
     * @ORM\Column(name="paddingTop", type="float", nullable=false)
     */
    private $paddingTop;

    /**
     * @var float
     *
     * @ORM\Column(name="paddingRight", type="float", nullable=false)
     */
    private $paddingRight;

    /**
     * @var float
     *
     * @ORM\Column(name="paddingBottom", type="float", nullable=false)
     */
    private $paddingBottom;

    /**
     * @var float
     *
     * @ORM\Column(name="paddingLeft", type="float", nullable=false)
     */
    private $paddingLeft;

    /**
     * @var string
     *
     * @ORM\Column(name="comment", type="text", nullable=true)
     */
    private $comment = null;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="BarcodeLabelPresetBlock", mappedBy="preset", cascade={"persist", "remove"})
     */
    private $blocks;

    public function __construct()
    {
        $this->blocks = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return mixed
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param string $type
     */
    public function setType($type)
    {
        $this->type = $type;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @param string $name
     */
    public function setName($name)
    {
        $this->name = $name;
    }

    /**
     * @return string
     */
    public function getPaperLayoutIdentifier()
    {
        return $this->paperLayoutIdentifier;
    }

    /**
     * @param string $paperLayoutIdentifier
     */
    public function setPaperLayoutIdentifier($paperLayoutIdentifier)
    {
        $this->paperLayoutIdentifier = $paperLayoutIdentifier;
    }

    /**
     * @return string
     */
    public function getTemplateIdentifier()
    {
        return $this->templateIdentifier;
    }

    /**
     * @param string $templateIdentifier
     */
    public function setTemplateIdentifier($templateIdentifier)
    {
        $this->templateIdentifier = $templateIdentifier;
    }

    /**
     * @return float|null
     */
    public function getPaperHeight()
    {
        return $this->paperHeight;
    }

    /**
     * @param float|null $paperHeight
     */
    public function setPaperHeight($paperHeight)
    {
        $this->paperHeight = $paperHeight;
    }

    /**
     * @return float|null
     */
    public function getPaperWidth()
    {
        return $this->paperWidth;
    }

    /**
     * @param float|null $paperWidth
     */
    public function setPaperWidth($paperWidth)
    {
        $this->paperWidth = $paperWidth;
    }

    /**
     * @return float
     */
    public function getPaddingTop()
    {
        return $this->paddingTop;
    }

    /**
     * @param float $paddingTop
     */
    public function setPaddingTop($paddingTop)
    {
        $this->paddingTop = $paddingTop;
    }

    /**
     * @return float
     */
    public function getPaddingRight()
    {
        return $this->paddingRight;
    }

    /**
     * @param float $paddingRight
     */
    public function setPaddingRight($paddingRight)
    {
        $this->paddingRight = $paddingRight;
    }

    /**
     * @return float
     */
    public function getPaddingBottom()
    {
        return $this->paddingBottom;
    }

    /**
     * @param float $paddingBottom
     */
    public function setPaddingBottom($paddingBottom)
    {
        $this->paddingBottom = $paddingBottom;
    }

    /**
     * @return float
     */
    public function getPaddingLeft()
    {
        return $this->paddingLeft;
    }

    /**
     * @param float $paddingLeft
     */
    public function setPaddingLeft($paddingLeft)
    {
        $this->paddingLeft = $paddingLeft;
    }

    /**
     * @return string|null
     */
    public function getComment()
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     */
    public function setComment($comment)
    {
        $this->comment = $comment;
    }

    /**
     * @return ArrayCollection
     */
    public function getBlocks()
    {
        return $this->blocks;
    }

    /**
     * Updates all values with the given array.
     *
     * @param array $values
     */
    public function updateWithValues(array $values)
    {
        $paddingBottom = floatval($values['paddingBottom'] ?: 0);
        $this->setPaddingBottom($paddingBottom);

        $paddingLeft = floatval($values['paddingLeft'] ?: 0);
        $this->setPaddingLeft($paddingLeft);

        $paddingTop = floatval($values['paddingTop'] ?: 0);
        $this->setPaddingTop($paddingTop);

        $paddingRight = floatval($values['paddingRight'] ?: 0);
        $this->setPaddingRight($paddingRight);

        $paddingBottom = floatval($values['paddingBottom'] ?: 0);
        $this->setPaddingBottom($paddingBottom);

        $paddingLeft = floatval($values['paddingLeft'] ?: 0);
        $this->setPaddingLeft($paddingLeft);

        $paperLayoutIdentifier = $values['paperLayoutIdentifier'];
        $this->setPaperLayoutIdentifier($paperLayoutIdentifier);

        $paperWidth = floatval($values['paperWidth'] ?: null);
        $this->setPaperWidth($paperWidth);

        $paperHeight = floatval($values['paperHeight'] ?: null);
        $this->setPaperHeight($paperHeight);

        // Check that the paper layout is valid (otherwise an exception is thrown)
        BarcodeLabelGridPaperLayout::createLabelGridPaperLayoutForPreset($this);

        $type = $values['type'];
        if (!$type) {
            throw new InvalidArgumentException('Invalid preset type.');
        }
        $this->setType($type);

        $templateIdentifier = $values['templateIdentifier'];
        $this->setTemplateIdentifier($templateIdentifier);

        $name = $values['name'] ?: null;
        if (!$name) {
            throw new \InvalidArgumentException('Invalid preset name.');
        }
        $this->setName($name);

        $comment = $values['comment'] ?: null;
        $this->setComment($comment);
    }
}
