<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel;

use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;
use Shopware\Plugins\ViisonPickwareERP\Components\BarcodeLabel\BarcodeLabel;

/**
 * @ORM\Entity(repositoryClass="Repository")
 * @ORM\Table(name="pickware_erp_bin_location_barcode_labels")
 */
class BinLocationBarcodeLabel extends ModelEntity implements BarcodeLabel
{
    /**
     * @var int $id
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var int $binLocationId
     *
     * @ORM\Column(name="binLocationId", type="integer", nullable=false)
     */
    private $binLocationId;

    /**
     * OWNING SIDE
     *
     * @var BinLocation $binLocation the bin location to identify by the barcode label
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation")
     * @ORM\JoinColumn(name="binLocationId", referencedColumnName="id", nullable=false)
     */
    protected $binLocation;

    /**
     * @var int $quantity how many labels to print
     *
     * @ORM\Column(name="quantity", type="integer", nullable=false)
     */
    private $quantity;

    /**
     * @var \DateTime $added when the barcode label was marked for printing
     *
     * @ORM\Column(name="added", type="datetime", nullable=false)
     */
    private $added;

    /**
     * BarcodeLabel constructor.
     *
     * @param BinLocation $binLocation the bin location to print a barcode label for
     * @param int $quantity how many labels to print
     */
    public function __construct($binLocation, $quantity = 1)
    {
        $this->binLocation = $binLocation;
        $this->quantity = $quantity;
        $this->added = new \DateTime();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return BinLocation the bin location to print a barcode label for
     */
    public function getBinLocation()
    {
        return $this->binLocation;
    }

    /**
     * @param BinLocation $binLocation the bin location to print a barcode label for
     */
    public function setBinLocation(BinLocation $binLocation)
    {
        $this->binLocation = $binLocation;
    }

    /**
     * @return int how many barcode labels to print
     */
    public function getQuantity()
    {
        return $this->quantity;
    }

    /**
     * @param int $quantity how many barcode labels to print
     */
    public function setQuantity($quantity)
    {
        $this->quantity = $quantity;
    }

    /**
     * Adds the given quantity to the quantity of this label.
     *
     * @param int $quantity how many more barcode labels to print
     */
    public function addQuantity($quantity)
    {
        $this->quantity += $quantity;
    }

    /**
     * @return \DateTime when this barcode label was marked for printing
     */
    public function getAdded()
    {
        return $this->added;
    }

    /**
     * @param \DateTime $added when this barcode label was marked for printing
     */
    public function setAdded(\DateTime $added)
    {
        $this->added = $added;
    }
}
