<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\BarcodeLabel;

use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use Shopware\Components\Model\ModelRepository;

class Repository extends ModelRepository
{
    /**
     * Returns a doctrine database query for the retrieval of a list of barcode label presets.
     *
     * @param string $presetType
     * @return Query
     */
    public function getBarcodeLabelPresetListQuery($presetType = null)
    {
        $builder = $this->getEntityManager()->createQueryBuilder();
        $builder
            ->select(
                'preset',
                'block'
            )
            ->from(BarcodeLabelPreset::class, 'preset')
            ->leftJoin('preset.blocks', 'block');

        if ($presetType !== null) {
            $builder
                ->where('preset.type = :presetType')
                ->setParameter('presetType', $presetType);
        }

        return $builder->getQuery();
    }

    /**
     * Returns a doctrine database query for the retrieval of a list of article barcode labels.
     *
     * @param int $start (optional)
     * @param int $limit (optional)
     * @param array $sort (optional)
     * @param array $filter (optional)
     * @param array $ids (optional)
     * @return Query
     */
    public function getArticleBarcodeLabelListQuery($start = null, $limit = null, $sort = [], $filter = [], $ids = null)
    {
        $builder = $this->getEntityManager()->createQueryBuilder();
        $builder->select('barcodeLabel')
            ->from(ArticleBarcodeLabel::class, 'barcodeLabel', 'barcodeLabel.articleDetailId')
            ->leftJoin('barcodeLabel.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('articleDetail.attribute', 'articleAttribute')
            ->addOrderBy($sort)
            ->addFilter($filter);

        $this->addIdFilterIfApplicable($builder, $ids);
        $this->addPagination($builder, $start, $limit);

        return $builder->getQuery();
    }

    /**
     * Returns a doctrine database query for the retrieval of a list of bin location barcode labels.
     *
     * @param int $start (optional)
     * @param int $limit (optional)
     * @param array $sort (optional)
     * @param array $filter (optional)
     * @param array $ids (optional)
     * @return Query
     */
    public function getBinLocationBarcodeLabelListQuery($start = null, $limit = null, $sort = [], $filter = [], $ids = null)
    {
        $builder = $this->getEntityManager()->createQueryBuilder();
        $builder->select('barcodeLabel')
            ->from(BinLocationBarcodeLabel::class, 'barcodeLabel')
            ->leftJoin('barcodeLabel.binLocation', 'binLocation')
            ->leftJoin('binLocation.warehouse', 'warehouse')
            ->addOrderBy($sort)
            ->addFilter($filter);

        $this->addIdFilterIfApplicable($builder, $ids);
        $this->addPagination($builder, $start, $limit);

        return $builder->getQuery();
    }

    /**
     * Add a filter to only return barcode labels with the given $ids to a barcode label list query builder.
     *
     * @param QueryBuilder $builder
     * @param $ids
     */
    private function addIdFilterIfApplicable($builder, $ids)
    {
        if ($ids) {
            $builder->where('barcodeLabel.id IN (:ids)')
                ->setParameter('ids', $ids);
        }
    }

    /**
     * Add pagination parameters to a barcode label list query builder.
     *
     * @param QueryBuilder $builder
     * @param $start
     * @param $limit
     */
    private function addPagination($builder, $start, $limit)
    {
        if ($start !== null) {
            $builder->setFirstResult($start);
        }
        if ($limit !== null) {
            $builder->setMaxResults($limit);
        }
    }

    /**
     * Remove all Stock items corresponding to the given ArticleDetail-id
     *
     * @param int $articleDetailId
     */
    public function deleteByArticleDetailId($articleDetailId)
    {
        $dql = 'DELETE Shopware\\CustomModels\\ViisonPickwareERP\\BarcodeLabel\\ArticleBarcodeLabel articleBarcodeLabel
                WHERE articleBarcodeLabel.articleDetailId = :articleDetailId';

        $this->getEntityManager()
            ->createQuery($dql)
            ->setParameter('articleDetailId', $articleDetailId)
            ->execute();
    }

    /**
     * Remove all Stock items corresponding to the given ArticleDetail-id
     *
     * @param int $binLocationId
     */
    public function deleteByBinLocationId($binLocationId)
    {
        $dql = 'DELETE Shopware\\CustomModels\\ViisonPickwareERP\\BarcodeLabel\\BinLocationBarcodeLabel binLocationBarcodeLabel
                WHERE binLocationBarcodeLabel.binLocationId = :binLocationId';

        $this->getEntityManager()
            ->createQuery($dql)
            ->setParameter('binLocationId', $binLocationId)
            ->execute();
    }
}
