<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\RestApi;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;

/**
 * @ORM\Entity
 * @ORM\Table(name="pickware_erp_rest_api_idempotent_operations")
 */
class RestApiIdempotentOperation extends ModelEntity
{
    const RECOVERY_POINT_STARTED = 'started';
    const RECOVERY_POINT_FINISHED = 'finished';

    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var DateTime
     *
     * @ORM\Column(name="createdAt", type="datetime", nullable=false)
     */
    private $createdAt;

    /**
     * @var string
     *
     * @ORM\Column(name="idempotencyKey", type="string", length=255, nullable=false)
     */
    private $idempotencyKey;

    /**
     * @var string
     *
     * @ORM\Column(name="controllerName", type="string", length=1000, nullable=false)
     */
    private $controllerName;

    /**
     * @var string
     *
     * @ORM\Column(name="actionName", type="string", length=1000, nullable=false)
     */
    private $actionName;

    /**
     * @var string
     *
     * @ORM\Column(name="recoveryPoint", type="string", length=255, nullable=false)
     */
    private $recoveryPoint;

    /**
     * @var int|null
     *
     * @ORM\Column(name="responseCode", type="integer", nullable=true)
     */
    private $responseCode;

    /**
     * @var string|null
     *
     * @ORM\Column(name="responseData", type="text", nullable=true)
     */
    private $responseData;

    /**
     * @var string|null
     *
     * @ORM\Column(name="intermediateState", type="text", nullable=true)
     */
    private $intermediateState;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\RestApi\RestApiRequest", mappedBy="idempotentOperation")
     */
    private $requests;

    /**
     * @var string
     *
     * @ORM\Column(name="initialRequestRawBodyChecksum", type="string", nullable=false)
     */
    private $initialRequestRawBodyChecksum;

    /**
     * @param string $idempotencyKey
     * @param \Enlight_Controller_Request_RequestHttp $request
     */
    public function __construct($idempotencyKey, \Enlight_Controller_Request_RequestHttp $request)
    {
        $this->createdAt = new DateTime();
        $this->idempotencyKey = $idempotencyKey;
        $this->controllerName = $request->getControllerName();
        $this->actionName = $request->getActionName();
        $this->initialRequestRawBodyChecksum = self::calculateRequestRawBodyChecksum($request);
        $this->recoveryPoint = self::RECOVERY_POINT_STARTED;
        $this->requests = new ArrayCollection();
    }

    /**
     * @param \Enlight_Controller_Request_RequestHttp $request
     * @return string
     */
    private static function calculateRequestRawBodyChecksum(\Enlight_Controller_Request_RequestHttp $request)
    {
        return sha1($request->getRawBody());
    }

    /**
     * @param \Enlight_Controller_Request_RequestHttp $request
     * @return bool
     */
    public function originRequestHasSameRawBodyAsRequest(\Enlight_Controller_Request_RequestHttp $request)
    {
        return $this->initialRequestRawBodyChecksum === self::calculateRequestRawBodyChecksum($request);
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return DateTime
     */
    public function getCreatedAt()
    {
        return $this->createdAt;
    }

    /**
     * @return string
     */
    public function getIdempotencyKey()
    {
        return $this->idempotencyKey;
    }

    /**
     * @return string
     */
    public function getControllerName()
    {
        return $this->controllerName;
    }

    /**
     * @return string
     */
    public function getActionName()
    {
        return $this->actionName;
    }

    /**
     * @return string
     */
    public function getRecoveryPoint()
    {
        return $this->recoveryPoint;
    }

    /**
     * @param string $recoveryPoint
     */
    public function setRecoveryPoint($recoveryPoint)
    {
        $this->recoveryPoint = $recoveryPoint;
    }

    /**
     * @return int|null
     */
    public function getResponseCode()
    {
        return $this->responseCode;
    }

    /**
     * @param int|null $responseCode
     */
    public function setResponseCode($responseCode)
    {
        $this->responseCode = $responseCode;
    }

    /**
     * @return array|string|null
     */
    public function getResponseData()
    {
        return $this->responseData;
    }

    /**
     * @param array|string|null $responseData
     */
    public function setResponseData($responseData)
    {
        $this->responseData = (is_array($responseData)) ? json_encode($responseData) : $responseData;
    }

    /**
     * @return array|null
     */
    public function getDecodedResponseData()
    {
        return (is_string($this->responseData)) ? json_decode($this->responseData, true) : null;
    }

    /**
     * @return ArrayCollection
     */
    public function getRequests()
    {
        return $this->requests;
    }

    /**
     * @param string|null $intermediateState
     */
    public function setIntermediateState($intermediateState = null)
    {
        $this->intermediateState = $intermediateState;
    }

    /**
     * @return string|null
     */
    public function getIntermediateState()
    {
        return $this->intermediateState;
    }

    /**
     * @return string
     */
    public function getInitialRequestRawBodyChecksum()
    {
        return $this->initialRequestRawBodyChecksum;
    }
}
