<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\ReturnShipment;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Order\Document\Document;
use Shopware\Models\Order\Order;
use Shopware\Models\User\User;

/**
 * @ORM\MappedSuperclass
 */
abstract class AbstractReturnShipment extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Id
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="number", type="string", nullable=false)
     */
    private $number;

    /**
     * @var int
     *
     * @ORM\Column(name="orderId", type="integer", nullable=false)
     */
    private $orderId;

    /**
     * @var int
     *
     * @ORM\Column(name="userId", type="integer", nullable=true)
     */
    private $userId = null;

    /**
     * @var int
     *
     * @ORM\Column(name="statusId", type="integer", nullable=false)
     */
    private $statusId;

    /**
     * @var int|null
     *
     * @ORM\Column(name="targetWarehouseId", type="integer", nullable=true)
     */
    private $targetWarehouseId = null;

    /**
     * @var int|null
     *
     * @ORM\Column(name="documentId", type="integer", nullable=true)
     */
    private $documentId = null;

    /**
     * @var DateTime
     *
     * @ORM\Column(name="created", type="datetime", nullable=false)
     */
    private $created;

    /**
     * OWNING SIDE
     *
     * @var Order
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Order\Order")
     * @ORM\JoinColumn(name="orderId", referencedColumnName="id")
     */
    private $order;

    /**
     * OWNING SIDE
     *
     * @var User
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\User\User")
     * @ORM\JoinColumn(name="userId", referencedColumnName="id")
     */
    private $user = null;

    /**
     * OWNING SIDE
     *
     * @var ReturnShipmentStatus
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentStatus")
     * @ORM\JoinColumn(name="statusId", referencedColumnName="id")
     */
    private $status;

    /**
     * OWNING SIDE
     *
     * @var Warehouse|null
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse")
     * @ORM\JoinColumn(name="targetWarehouseId", referencedColumnName="id")
     */
    private $targetWarehouse = null;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem", mappedBy="returnShipment", cascade={"all"})
     */
    protected $items;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentAttachment", mappedBy="returnShipment", cascade={"all"})
     */
    protected $attachments;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     * @ORM\OneToMany(targetEntity="ReturnShipmentInternalComment", mappedBy="returnShipment", cascade={"all"})
     */
    protected $internalComments;

    /**
     * @var Document|null
     *
     * @ORM\OneToOne(targetEntity="Shopware\Models\Order\Document\Document")
     * @ORM\JoinColumn(name="documentId", referencedColumnName="id")
     */
    protected $document = null;

    /**
     * @param Order $order
     * @param ReturnShipmentStatus $status
     */
    public function __construct(Order $order, $number, ReturnShipmentStatus $status)
    {
        $this->created = new DateTime();
        $this->number = $number;
        $this->order = $order;
        $this->status = $status;
        $this->items = new ArrayCollection();
        $this->attachments = new ArrayCollection();
        $this->internalComments = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return DateTime
     */
    public function getCreated()
    {
        return $this->created;
    }

    /**
     * @param DateTime|string $created
     */
    public function setCreated($created)
    {
        if ($created === null) {
            return;
        }

        $this->created = ($created instanceof DateTime) ? $created : new DateTime($created);
    }

    /**
     * @return Order
     */
    public function getOrder()
    {
        return $this->order;
    }

    /**
     * @param Order $order
     */
    public function setOrder(Order $order)
    {
        $this->order = $order;
    }

    /**
     * @return User
     */
    public function getUser()
    {
        return $this->user;
    }

    /**
     * @param User|null $user
     */
    public function setUser(User $user = null)
    {
        $this->user = $user;
    }

    /**
     * @return ReturnShipmentStatus
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * @param ReturnShipmentStatus $status
     */
    public function setStatus(ReturnShipmentStatus $status)
    {
        $this->status = $status;
    }

    /**
     * @param ReturnShipmentStatus $status
     */
    public function setStatusToAtLeast(ReturnShipmentStatus $status)
    {
        if ($status->getProcessStep() <= $this->status->getProcessStep()) {
            return;
        }
        $this->setStatus($status);
    }

    /**
     * @param ReturnShipmentStatus $status
     */
    public function setStatusToAtMost(ReturnShipmentStatus $status)
    {
        if ($status->getProcessStep() >= $this->status->getProcessStep()) {
            return;
        }
        $this->setStatus($status);
    }

    /**
     * @return Warehouse|null
     */
    public function getTargetWarehouse()
    {
        return $this->targetWarehouse;
    }

    /**
     * @param Warehouse|null $targetWarehouse
     */
    public function setTargetWarehouse(Warehouse $targetWarehouse = null)
    {
        $this->targetWarehouse = $targetWarehouse;
    }

    /**
     * @return ArrayCollection
     */
    public function getItems()
    {
        return $this->items;
    }

    /**
     * @param ReturnShipmentItem $item
     */
    public function addItem(ReturnShipmentItem $item)
    {
        $this->items->add($item);
    }

    /**
     * @param ReturnShipmentItem $item
     */
    public function removeItem(ReturnShipmentItem $item)
    {
        if ($this->items->contains($item)) {
            $this->items->removeElement($item);
        }
    }

    /**
     * @return ArrayCollection
     */
    public function getAttachments()
    {
        return $this->attachments;
    }

    /**
     * @param ReturnShipmentAttachment $attachment
     */
    public function addAttachment(ReturnShipmentAttachment $attachment)
    {
        $this->attachments->add($attachment);
    }

    /**
     * @param ReturnShipmentAttachment $attachment
     */
    public function removeAttachment(ReturnShipmentAttachment $attachment)
    {
        if ($this->attachments->contains($attachment)) {
            $this->attachments->removeElement($attachment);
        }
    }

    /**
     * @return ArrayCollection
     */
    public function getInternalComments()
    {
        return $this->internalComments;
    }

    /**
     * @param ReturnShipmentInternalComment $internalComment
     */
    public function addInternalComment(ReturnShipmentInternalComment $internalComment)
    {
        $this->internalComments->add($internalComment);
    }

    /**
     * @param ReturnShipmentInternalComment $internalComment
     */
    public function removeInternalComment(ReturnShipmentInternalComment $internalComment)
    {
        if ($this->internalComments->contains($internalComment)) {
            $this->internalComments->removeElement($internalComment);
        }
    }

    /**
     * @return string
     */
    public function getNumber()
    {
        return $this->number;
    }

    /**
     * @param string $number
     */
    public function setNumber($number)
    {
        $this->number = $number;
    }

    /**
     * @return Document|null
     */
    public function getDocument()
    {
        return $this->document;
    }

    /**
     * @param Document|null $document
     */
    public function setDocument(Document $document = null)
    {
        $this->document = $document;
    }
}
