<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\ReturnShipment;

use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\Models\Order\Detail as OrderDetail;

/**
 * @ORM\Entity(repositoryClass="ReturnShipmentRepository")
 * @ORM\Table(name="pickware_erp_return_shipment_items")
 * @ORM\HasLifecycleCallbacks
 */
class ReturnShipmentItem extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Id
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var int
     *
     * @ORM\Column(name="returnShipmentId", type="integer", nullable=false)
     */
    private $returnShipmentId;

    /**
     * @var int
     *
     * @ORM\Column(name="orderDetailId", type="integer", nullable=false)
     */
    private $orderDetailId;

    /**
     * @var int
     *
     * @ORM\Column(name="returnedQuantity", type="integer", nullable=false)
     */
    private $returnedQuantity = 0;

    /**
     * @var int
     *
     * @ORM\Column(name="writtenOffQuantity", type="integer", nullable=false)
     */
    private $writtenOffQuantity = 0;

    /**
     * @var int
     *
     * @ORM\Column(name="cancelledQuantity", type="integer", nullable=false)
     */
    private $cancelledQuantity = 0;

    /**
     * OWNING SIDE
     *
     * @var ReturnShipment
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipment", inversedBy="items")
     * @ORM\JoinColumn(name="returnShipmentId", referencedColumnName="id")
     */
    protected $returnShipment;

    /**
     * OWNING SIDE
     *
     * @var OrderDetail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Order\Detail")
     * @ORM\JoinColumn(name="orderDetailId", referencedColumnName="id")
     */
    protected $orderDetail;

    /**
     * @var int
     */
    private $deltaReturnedQuantity = 0;

    /**
     * @var int
     */
    private $deltaWrittenOffQuantity = 0;

    /**
     * @param ReturnShipment $returnShipment
     * @param OrderDetail $orderDetail
     */
    public function __construct(ReturnShipment $returnShipment, OrderDetail $orderDetail)
    {
        $this->returnShipment = $returnShipment;
        $this->orderDetail = $orderDetail;
        $this->returnShipment->addItem($this);
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getReturnedQuantity()
    {
        return $this->returnedQuantity;
    }

    /**
     * @param int $returnedQuantity
     */
    public function setReturnedQuantity($returnedQuantity)
    {
        $this->deltaReturnedQuantity += $returnedQuantity - $this->returnedQuantity;

        $this->returnedQuantity = $returnedQuantity;
    }

    /**
     * @return int
     */
    public function getWrittenOffQuantity()
    {
        return $this->writtenOffQuantity;
    }

    /**
     * @param int $writtenOffQuantity
     */
    public function setWrittenOffQuantity($writtenOffQuantity)
    {
        $this->deltaWrittenOffQuantity += $writtenOffQuantity - $this->writtenOffQuantity;

        $this->writtenOffQuantity = $writtenOffQuantity;
    }

    /**
     * @return ReturnShipment
     */
    public function getReturnShipment()
    {
        return $this->returnShipment;
    }

    /**
     * @return OrderDetail
     */
    public function getOrderDetail()
    {
        return $this->orderDetail;
    }

    /**
     * Returns the delta of returnedQuantity made since the last load of the model or call of resetDeltas().
     *
     * @return int
     */
    public function getDeltaReturnedQuantity()
    {
        return $this->deltaReturnedQuantity;
    }

    /**
     * Returns the delta of writtenOffQuantity made since the last load of the model or call of resetDeltas().
     *
     * @return int
     */
    public function getDeltaWrittenOffQuantity()
    {
        return $this->deltaWrittenOffQuantity;
    }

    /**
     * @return int
     */
    public function getCancelledQuantity()
    {
        return $this->cancelledQuantity;
    }

    /**
     * @param int $cancelledQuantity
     */
    public function setCancelledQuantity($cancelledQuantity)
    {
        $this->cancelledQuantity = $cancelledQuantity;
    }

    /**
     * Sets the deltas of writtenOffQuantity and returnedQuantity to 0.
     *
     * @ORM\PostLoad
     */
    public function resetDeltas()
    {
        $this->deltaReturnedQuantity = 0;
        $this->deltaWrittenOffQuantity = 0;
    }
}
