<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\ReturnShipment;

use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Join;
use Shopware\Components\Model\ModelRepository;
use Shopware\Components\Model\QueryBuilder;
use Shopware\Models\Order\Order;

class ReturnShipmentRepository extends ModelRepository
{
    /**
     * @param array $sort
     * @param array $filter
     * @param int $offset
     * @param int $limit
     * @return Query
     */
    public function getReturnShipmentListQuery(array $sort = [], array $filter = [], $offset = null, $limit = null)
    {
        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->getEntityManager()->createQueryBuilder();
        $queryBuilder
            ->setAlias('returnShipment')
            ->select(
                'returnShipment',
                'items',
                'orderDetail',
                'attachments',
                'media',
                'user'
            )
            ->from(ReturnShipment::class, 'returnShipment')
            ->leftJoin('returnShipment.items', 'items')
            ->leftJoin('items.orderDetail', 'orderDetail')
            ->leftJoin('returnShipment.attachments', 'attachments')
            ->leftJoin('attachments.media', 'media')
            ->leftJoin('returnShipment.user', 'user')
            ->addFilter($filter)
            ->addOrderBy($sort);

        if ($offset !== null) {
            $queryBuilder->setFirstResult($offset);
        }
        if ($limit !== null) {
            $queryBuilder->setMaxResults($limit);
        }

        return $queryBuilder->getQuery();
    }

    /**
     * @param Order $order
     * @return ReturnShipmentStatus[]
     */
    public function findAllReturnShipmentStatusOfOrder(Order $order)
    {
        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->getEntityManager()->createQueryBuilder();
        $queryBuilder
            ->select('status')
            ->from(ReturnShipmentStatus::class, 'status')
            ->leftJoin(ReturnShipment::class, 'returnShipment', Join::WITH, 'returnShipment.status = status')
            ->where('returnShipment.order = :order')
            ->setParameter('order', $order);

        return $queryBuilder->getQuery()->getResult();
    }

    /**
     * @return ReturnShipmentStatus
     */
    public function getInitialReturnShipmentStatus()
    {
        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->getEntityManager()->createQueryBuilder();
        $queryBuilder
            ->select('status')
            ->from(ReturnShipmentStatus::class, 'status')
            ->orderBy('status.processStep', 'ASC')
            ->setMaxResults(1);

        return $queryBuilder->getQuery()->getOneOrNullResult();
    }
}
