<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\StockLedger;

use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation;

/**
 * @ORM\Entity
 * @ORM\Table(name="pickware_erp_bin_location_stock_snapshots", uniqueConstraints={
 *      @ORM\UniqueConstraint(name="stockEntryId_binLocationId", columns={
 *          "stockLedgerEntryId",
 *          "binLocationId"
 *      })
 * })
 */
class BinLocationStockSnapshot extends ModelEntity
{
    /**
     * @var int $id
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var int
     *
     * @ORM\Column(name="stockLedgerEntryId", type="integer", nullable=false)
     */
    private $stockLedgerEntryId;

    /**
     * @var int $binLocationId
     *
     * @ORM\Column(name="binLocationId", type="integer", nullable=false)
     */
    private $binLocationId;

    /**
     * @var int $stock
     *
     * @ORM\Column(name="stock", type="integer", nullable=false)
     */
    private $stock;

    /**
     * OWNING SIDE
     *
     * @var StockLedgerEntry
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry", inversedBy="binLocationStockSnapshots")
     * @ORM\JoinColumn(name="stockLedgerEntryId", referencedColumnName="id", onDelete="CASCADE")
     */
    protected $stockLedgerEntry;

    /**
     * OWNING SIDE
     *
     * @var BinLocation $binLocation
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation", inversedBy="stockSnapshots")
     * @ORM\JoinColumn(name="binLocationId", referencedColumnName="id", onDelete="CASCADE")
     */
    protected $binLocation;

    /**
     * @param StockLedgerEntry $stockLedgerEntry
     * @param BinLocation $binLocation
     * @param int $stock
     */
    public function __construct(StockLedgerEntry $stockLedgerEntry, BinLocation $binLocation, $stock)
    {
        $this->stockLedgerEntry = $stockLedgerEntry;
        $this->stockLedgerEntry->getBinLocationStockSnapshots()->add($this);
        $this->binLocation = $binLocation;
        $this->binLocation->getStockSnapshots()->add($this);
        $this->stock = $stock;

        // Explicitly set the two IDs, which form the unique identifier of this entity, because Doctrine
        // won't do it, even though we have already set the associated entities
        $this->stockLedgerEntryId = $stockLedgerEntry->getId();
        $this->binLocationId = $binLocation->getId();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getStock()
    {
        return $this->stock;
    }

    /**
     * @param int $stock
     */
    public function setStock($stock)
    {
        $this->stock = $stock;
    }

    /**
     * @return StockLedgerEntry
     */
    public function getStockLedgerEntry()
    {
        return $this->stockLedgerEntry;
    }

    /**
     * @return BinLocation
     */
    public function getBinLocation()
    {
        return $this->binLocation;
    }
}
