<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\StockLedger;

use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping;
use Shopware\Models\Order\Detail as OrderDetail;

/**
 * @ORM\Entity
 * @ORM\Table(name="pickware_erp_order_stock_reservations", uniqueConstraints={
 *      @ORM\UniqueConstraint(name="binLocationArticleDetailId_orderDetailId", columns={
 *          "articleDetailBinLocationMappingId",
 *          "orderDetailId"
 *      })
 * })
 */
class OrderStockReservation extends ModelEntity
{
    /**
     * @var int $id
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var int
     *
     * @ORM\Column(name="articleDetailBinLocationMappingId", type="integer", nullable=false)
     */
    private $articleDetailBinLocationMappingId;

    /**
     * @var int $orderDetailId
     *
     * @ORM\Column(name="orderDetailId", type="integer", nullable=false)
     */
    private $orderDetailId;

    /**
     * @var int $quantity
     *
     * @ORM\Column(name="quantity", type="integer", nullable=false)
     */
    private $quantity;

    /**
     * OWNING SIDE
     *
     * @var ArticleDetailBinLocationMapping
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping", inversedBy="stockReservations")
     * @ORM\JoinColumn(name="articleDetailBinLocationMappingId", referencedColumnName="id", onDelete="CASCADE")
     */
    protected $articleDetailBinLocationMapping;

    /**
     * OWNING SIDE
     *
     * @var OrderDetail $orderDetail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Order\Detail")
     * @ORM\JoinColumn(name="orderDetailId", referencedColumnName="id", onDelete="CASCADE")
     */
    protected $orderDetail;

    /**
     * @param ArticleDetailBinLocationMapping $articleDetailBinLocationMapping
     * @param OrderDetail $orderDetail
     * @param int $quantity
     */
    public function __construct(ArticleDetailBinLocationMapping $articleDetailBinLocationMapping, OrderDetail $orderDetail, $quantity)
    {
        $this->articleDetailBinLocationMapping = $articleDetailBinLocationMapping;
        $this->articleDetailBinLocationMapping->getStockReservations()->add($this);
        $this->orderDetail = $orderDetail;
        $this->quantity = $quantity;

        // Explicitly set the FK ID, because Doctrine won't do it, even though we have already set the associated entity
        $this->articleDetailBinLocationMappingId = $articleDetailBinLocationMapping->getId();
        $this->orderDetailId = $orderDetail->getId();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return ArticleDetailBinLocationMapping
     */
    public function getArticleDetailBinLocationMapping()
    {
        return $this->articleDetailBinLocationMapping;
    }

    /**
     * @return OrderDetail
     */
    public function getOrderDetail()
    {
        return $this->orderDetail;
    }

    /**
     * @return int
     */
    public function getQuantity()
    {
        return $this->quantity;
    }

    /**
     * @param int $quantity
     */
    public function setQuantity($quantity)
    {
        $this->quantity = $quantity;
    }
}
