<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\StockLedger;

use Shopware\Components\Model\ModelRepository;

class StockItemRepository extends ModelRepository
{
    /**
     * Fetches the stock entry items of all sale stock entries associated with any of the given $orderDetailIds and
     * returns an array grouped by order detail ID of the stock entry items as arrays.
     *
     * @param int[] $orderDetailIds
     * @return array[]
     */
    public function getSaleStockItemsAsArrays(array $orderDetailIds)
    {
        if (count($orderDetailIds) === 0) {
            return [];
        }

        // Fetch all stock entry items for the given order detail IDs
        $builder = $this->getEntityManager()->createQueryBuilder();
        $builder
            ->select(
                'stockItem',
                'stockLedgerEntry',
                'propertyValues',
                'itemProperty'
            )
            ->from(StockItem::class, 'stockItem')
            ->leftJoin('stockItem.stockLedgerEntry', 'stockLedgerEntry')
            ->leftJoin('stockItem.propertyValues', 'propertyValues')
            ->leftJoin('propertyValues.itemProperty', 'itemProperty')
            ->where('stockLedgerEntry.orderDetailId IN (:orderDetailIds)')
            ->andWhere('stockLedgerEntry.type = :typeSale')
            ->setParameters([
                'orderDetailIds' => $orderDetailIds,
                'typeSale' => StockLedgerEntry::TYPE_SALE,
            ]);
        $result = $builder->getQuery()->getArrayResult();

        // Extract the stock entry items and group them by article detail
        $groupedResult = [];
        foreach ($result as $row) {
            $orderDetailId = $row['stockLedgerEntry']['orderDetailId'];
            if (!isset($groupedResult[$orderDetailId])) {
                $groupedResult[$orderDetailId] = [];
            }
            unset($row['stockLedgerEntry']);
            $groupedResult[$orderDetailId][] = $row;
        }

        return $groupedResult;
    }
}
