<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\StockLedger;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItem;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;

/**
 * @ORM\Entity(repositoryClass="StockLedgerEntryRepository")
 * @ORM\Table(name="pickware_erp_stock_ledger_entries")
 */
class StockLedgerEntry extends ModelEntity
{
    const TYPE_PURCHASE = 'purchase';
    const TYPE_SALE = 'sale';
    const TYPE_RETURN = 'return';
    const TYPE_STOCKTAKE = 'stocktake';
    const TYPE_MANUAL = 'manual';
    const TYPE_INCOMING = 'incoming';
    const TYPE_OUTGOING = 'outgoing';
    const TYPE_RELOCATION = 'relocation';
    const TYPE_INITIALIZATION = 'initialization';
    const TYPE_WRITE_OFF = 'write-off';

    /**
     * @var integer $id
     *
     * @ORM\Id
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var string $type
     *
     * @ORM\Column(name="type", type="string", nullable=true)
     */
    private $type;

    /**
     * @var int $oldStock
     *
     * @ORM\Column(name="oldStock", type="integer", nullable=false)
     */
    private $oldStock;

    /**
     * @var int
     *
     * @ORM\Column(name="newStock", type="integer", nullable=false)
     */
    private $newStock;

    /**
     * @var int $changeAmount
     *
     * @ORM\Column(name="changeAmount", type="integer", nullable=true)
     */
    private $changeAmount;

    /**
     * @var float $purchasePrice
     *
     * @ORM\Column(name="purchasePrice", type="decimal", precision=10, scale=2, nullable=true)
     */
    private $purchasePrice;

    /**
     * @var string $comment
     *
     * @ORM\Column(name="comment", type="string", nullable=true)
     */
    private $comment;

    /**
     * @var DateTime $created
     *
     * @ORM\Column(name="created", type="datetime", nullable=false)
     */
    private $created;

    /**
     * @var bool True if this entry is a correction for it's type. This means it's change-amount is against its types
     *      natural direction.
     *
     * @ORM\Column(name="correction", type="boolean", nullable=false)
     */
    private $correction = false;

    /**
     * @var integer $sourceLotEntryId
     * @ORM\Column(name="sourceLotEntryId", type="integer", nullable=true)
     */
    private $sourceLotEntryId;

    /**
     * OWNING SIDE
     *
     * @var StockLedgerEntry
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry", inversedBy="lotAssignedEntries", cascade={"persist"})
     * @ORM\JoinColumn(name="sourceLotEntryId", nullable=true, referencedColumnName="id", onDelete="CASCADE")
     */
    private $sourceLotEntry;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry", mappedBy="sourceLotEntry", cascade={"all"}))
     * @ORM\OrderBy({"id" = "ASC"})
     */
    private $lotAssignedEntries;

    /**
     * @var integer $articleDetailId
     * @ORM\Column(name="articleDetailId", type="integer", nullable=false, options={"unsigned"=true})
     */
    private $articleDetailId;

    /**
     * @var \Shopware\Models\Article\Detail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Article\Detail")
     * @ORM\JoinColumn(name="articleDetailId", referencedColumnName="id", onDelete="CASCADE")
     */
    private $articleDetail;

    /**
     * @var integer $orderDetailId
     * @ORM\Column(name="orderDetailId", type="integer", nullable=true)
     */
    private $orderDetailId;

    /**
     * @var \Shopware\Models\Order\Detail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Order\Detail")
     * @ORM\JoinColumn(name="orderDetailId", referencedColumnName="id", onDelete="SET NULL")
     */
    private $orderDetail;

    /**
     * @var integer $userId
     * @ORM\Column(name="userId", type="integer", nullable=true)
     */
    private $userId;

    /**
     * @var \Shopware\Models\User\User
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\User\User")
     * @ORM\JoinColumn(name="userId", referencedColumnName="id", onDelete="SET NULL")
     */
    private $user;

    /**
     * @var integer $supplierOrderItemId
     *
     * @ORM\Column(name="supplierOrderItemId", type="integer", nullable=true)
     */
    private $supplierOrderItemId;

    /**
     * @var SupplierOrderItem
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItem")
     * @ORM\JoinColumn(name="supplierOrderItemId", referencedColumnName="id", onDelete="SET NULL")
     */
    protected $supplierOrderItem;

    /**
     * @var integer $returnShipmentItemId
     *
     * @ORM\Column(name="returnShipmentItemId", type="integer", nullable=true)
     */
    private $returnShipmentItemId;

    /**
     * OWNING SIDE
     *
     * @var ReturnShipmentItem
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem")
     * @ORM\JoinColumn(name="returnShipmentItemId", referencedColumnName="id", onDelete="SET NULL")
     */
    protected $returnShipmentItem;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\StockLedger\BinLocationStockSnapshot", mappedBy="stockLedgerEntry", cascade={"persist"}))
     */
    protected $binLocationStockSnapshots;

    /**
     * @var int $warehouseId
     *
     * @ORM\Column(name="warehouseId", type="integer", nullable=false)
     */
    private $warehouseId;

    /**
     * @var int $warehouseId
     *
     * @ORM\Column(name="transactionId", type="string", nullable=false)
     */
    private $transactionId;

    /**
     * OWNING SIDE
     *
     * @var Warehouse $warehouse
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse")
     * @ORM\JoinColumn(name="warehouseId", referencedColumnName="id", onDelete="CASCADE")
     */
    protected $warehouse;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockItem", mappedBy="stockLedgerEntry")
     */
    protected $stockItems;

    public function __construct()
    {
        $this->created = new DateTime();
        $this->lotAssignedEntries = new ArrayCollection();
        $this->binLocationStockSnapshots = new ArrayCollection();
        $this->stockItems = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param string $type
     */
    public function setType($type)
    {
        if (!in_array($type, [self::TYPE_PURCHASE, self::TYPE_SALE, self::TYPE_RETURN, self::TYPE_STOCKTAKE, self::TYPE_MANUAL, self::TYPE_INCOMING, self::TYPE_OUTGOING, self::TYPE_RELOCATION, self::TYPE_INITIALIZATION, self::TYPE_WRITE_OFF])) {
            throw new \InvalidArgumentException('Invalid type');
        }
        $this->type = $type;
    }

    /**
     * @return int
     */
    public function getOldStock()
    {
        return $this->oldStock;
    }

    /**
     * @param int $oldStock
     */
    public function setOldStock($oldStock)
    {
        $this->oldStock = $oldStock;
        $this->newStock = $this->oldStock + $this->getChangeAmount();
    }

    /**
     * @return int
     */
    public function getNewStock()
    {
        return $this->newStock;
    }

    /**
     * @return integer
     */
    public function getChangeAmount()
    {
        return $this->changeAmount;
    }

    /**
     * @param integer $changeAmount
     */
    public function setChangeAmount($changeAmount)
    {
        $this->changeAmount = $changeAmount;
        $this->newStock = $this->getOldStock() + $this->changeAmount;
    }

    /**
     * @return float|null
     */
    public function getPurchasePrice()
    {
        return $this->purchasePrice;
    }

    /**
     * @param float|null $purchasePrice
     */
    public function setPurchasePrice($purchasePrice)
    {
        $this->purchasePrice = $purchasePrice;
    }

    /**
     * @return string
     */
    public function getComment()
    {
        return $this->comment;
    }

    /**
     * @param string $comment
     */
    public function setComment($comment)
    {
        $this->comment = $comment;
    }

    /**
     * @return DateTime
     */
    public function getCreated()
    {
        return $this->created;
    }

    /**
     * @param DateTime $created
     */
    public function setCreated(DateTime $created)
    {
        $this->created = $created;
    }

    /**
     * @return bool
     */
    public function isCorrection()
    {
        return $this->correction;
    }

    /**
     * @param bool $isCorrection
     */
    public function setCorrection($isCorrection)
    {
        $this->correction = $isCorrection;
    }

    /**
     * @param StockLedgerEntry|null $sourceLotEntry
     */
    public function setSourceLotEntry(StockLedgerEntry $sourceLotEntry = null)
    {
        $this->sourceLotEntry = $sourceLotEntry;
    }

    /**
     * @return StockLedgerEntry
     */
    public function getSourceLotEntry()
    {
        return $this->sourceLotEntry;
    }

    /**
     * @return ArrayCollection
     */
    public function getLotAssignedEntries()
    {
        return $this->lotAssignedEntries;
    }

    /**
     * @param \Shopware\Models\Article\Detail|null $articleDetail
     */
    public function setArticleDetail(\Shopware\Models\Article\Detail $articleDetail = null)
    {
        $this->articleDetail = $articleDetail;
    }

    /**
     * @return \Shopware\Models\Article\Detail
     */
    public function getArticleDetail()
    {
        return $this->articleDetail;
    }

    /**
     * @param \Shopware\Models\Order\Detail|null $orderDetail
     */
    public function setOrderDetail(\Shopware\Models\Order\Detail $orderDetail = null)
    {
        $this->orderDetail = $orderDetail;
    }

    /**
     * @return \Shopware\Models\Order\Detail
     */
    public function getOrderDetail()
    {
        return $this->orderDetail;
    }

    /**
     * @param \Shopware\Models\User\User|null $user
     */
    public function setUser(\Shopware\Models\User\User $user = null)
    {
        $this->user = $user;
    }

    /**
     * @return \Shopware\Models\User\User
     */
    public function getUser()
    {
        return $this->user;
    }

    /**
     * @return SupplierOrderItem
     */
    public function getSupplierOrderItem()
    {
        return $this->supplierOrderItem;
    }

    /**
     * @param SupplierOrderItem|null $supplierOrderItem
     */
    public function setSupplierOrderItem(SupplierOrderItem $supplierOrderItem = null)
    {
        $this->supplierOrderItem = $supplierOrderItem;
    }

    /**
     * @return ReturnShipmentItem
     */
    public function getReturnShipmentItem()
    {
        return $this->returnShipmentItem;
    }

    /**
     * @param ReturnShipmentItem|null $returnShipmentItem
     */
    public function setReturnShipmentItem(ReturnShipmentItem $returnShipmentItem = null)
    {
        $this->returnShipmentItem = $returnShipmentItem;
    }

    /**
     * @return ArrayCollection
     */
    public function getBinLocationStockSnapshots()
    {
        return $this->binLocationStockSnapshots;
    }

    /**
     * @return Warehouse
     */
    public function getWarehouse()
    {
        return $this->warehouse;
    }

    /**
     * @param Warehouse $warehouse
     */
    public function setWarehouse(Warehouse $warehouse)
    {
        $this->warehouse = $warehouse;
    }

    /**
     * @return string
     */
    public function getTransactionId()
    {
        return $this->transactionId;
    }

    /**
     * @param string $transactionId
     */
    public function setTransactionId($transactionId)
    {
        $this->transactionId = $transactionId;
    }

    /**
     * @return ArrayCollection
     */
    public function getStockItems()
    {
        return $this->stockItems;
    }
}
