<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\StockLedger;

use Shopware\Components\Model\ModelRepository;

class StockLedgerEntryRepository extends ModelRepository
{
    /**
     * Creates and returns a query, which fetchs extended data of all stock entities
     * and applies pagination, sorting and filters if given.
     *
     * @param int $start
     * @param int $limit
     * @param array $sort
     * @param array $filter
     * @return \Doctrine\ORM\Query
     */
    public function getAllStocksQuery($start = 0, $limit = null, array $sort = [], array $filter = [])
    {
        $builder = $this->getEntityManager()->createQueryBuilder();
        $builder->select(
            'stockLedgerEntry.id AS id',
            'stockLedgerEntry.articleDetailId AS articleDetailId',
            'stockLedgerEntry.warehouseId AS warehouseId',
            'stockLedgerEntry.type AS type',
            'stockLedgerEntry.correction AS correction',
            'stockLedgerEntry.changeAmount AS changeAmount',
            'stockLedgerEntry.oldStock AS oldStock',
            'stockLedgerEntry.newStock AS newStock',
            'stockLedgerEntry.purchasePrice AS purchasePrice',
            'stockLedgerEntry.created AS created',
            'stockLedgerEntry.comment AS comment',
            'articleDetail.number AS articleNumber',
            'article.id AS articleId',
            'article.name AS articleName',
            'user.name AS username',
            'orderDetail.orderId AS orderId',
            'supplierOrderItem.supplierOrderId AS supplierOrderId',
            'returnShipmentItem.returnShipmentId AS returnShipmentId'
        )->from(StockLedgerEntry::class, 'stockLedgerEntry')
            ->leftJoin('stockLedgerEntry.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('stockLedgerEntry.user', 'user')
            ->leftJoin('stockLedgerEntry.supplierOrderItem', 'supplierOrderItem')
            ->leftJoin('stockLedgerEntry.orderDetail', 'orderDetail')
            ->leftJoin('stockLedgerEntry.returnShipmentItem', 'returnShipmentItem')
            ->addFilter($filter)
            ->addOrderBy($sort);

        // Add pagination
        if ($start !== null) {
            $builder->setFirstResult($start);
        }
        if ($limit !== null) {
            $builder->setMaxResults($limit);
        }

        return $builder->getQuery();
    }

    /**
     * Creates a query to fetch the total count of stock entities matching the given filter.
     *
     * @param array $filter
     * @return \Doctrine\ORM\Query
     */
    public function getAllStocksTotalCountQuery(array $filter = [])
    {
        $builder = $this->getEntityManager()->createQueryBuilder();
        $builder->select('COUNT(stockLedgerEntry)')
            ->from(StockLedgerEntry::class, 'stockLedgerEntry')
            ->addFilter($filter);

        return $builder->getQuery();
    }

    /**
     * Aggregated filtered stock entries and returns the a sorted result array
     * containing the aggregation results as well as the info about the respective article details.
     *
     * @param array $filter
     * @param array $sort
     * @return array
     */
    public function getInventory($filter = [], $sort = [])
    {
        $builder = $this->getEntityManager()->createQueryBuilder();
        $builder->select(
            'article.id AS articleId',
            'articleDetail.id AS articleDetailId',
            'articleDetail.number AS ordernumber',
            'article.name AS articlename',
            'articleDetail.additionalText AS additionalText',
            'SUM(stockLedgerEntry.changeAmount) AS instock',
            'tax.tax AS taxrate'
        )
            ->from(StockLedgerEntry::class, 'stockLedgerEntry')
            ->leftJoin('stockLedgerEntry.articleDetail', 'articleDetail')
            ->leftJoin('articleDetail.article', 'article')
            ->leftJoin('articleDetail.attribute', 'attribute')
            ->leftJoin('article.tax', 'tax')
            ->where('article.esds IS EMPTY')
            ->andWhere('attribute.pickwareStockManagementDisabled != 1')
            ->addFilter($filter)
            ->addOrderBy($sort)
            ->groupBy('articleDetail.id');

        // Select the summed purchase price with respect to the purchase price mode
        $purchasePriceMode = Shopware()->Container()->get('plugins')->get('Core')->get('ViisonPickwareERP')->Config(
        )->get('purchasePriceMode');
        if ($purchasePriceMode === 'net') {
            $builder->addSelect('SUM(stockLedgerEntry.changeAmount * stockLedgerEntry.purchasePrice) AS net')
                ->addSelect('SUM((stockLedgerEntry.changeAmount * stockLedgerEntry.purchasePrice) * (1.0 + tax.tax / 100.0)) AS gross');
        } else {
            $builder->addSelect('SUM((stockLedgerEntry.changeAmount * stockLedgerEntry.purchasePrice) / (1.0 + tax.tax / 100.0)) AS net')
                ->addSelect('SUM(stockLedgerEntry.changeAmount * stockLedgerEntry.purchasePrice) AS gross');
        }

        return $builder->getQuery()->getArrayResult();
    }
}
