<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\StockValuation;

use DateTime;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use InvalidArgumentException;
use LogicException;
use Shopware\Components\Model\ModelEntity;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Plugins\ViisonPickwareERP\Components\StockValuation\StockValuationException;

/**
 * @ORM\Entity
 * @ORM\Table(name="pickware_erp_stock_valuation_reports")
 */
class Report extends ModelEntity
{
    const METHOD_AVERAGE = 'average';
    const METHOD_FIFO = 'fifo';
    const METHOD_LIFO = 'lifo';

    const METHODS = [
        self::METHOD_AVERAGE,
        self::METHOD_FIFO,
        self::METHOD_LIFO,
    ];

    const GENERATION_STEP_REPORT_CREATED = 'report-created';
    const GENERATION_STEP_REPORT_PREPARED = 'report-prepared';
    const GENERATION_STEP_STOCKS_CALCULATED = 'stocks-calculated';
    const GENERATION_STEP_PURCHASES_CALCULATED = 'purchases-calculated';
    const GENERATION_STEP_AVERAGE_PURCHASE_PRICE_CALCULATED = 'average-purchase-price-calculated';
    const GENERATION_STEP_STOCK_RATED = 'stock-rated';
    const GENERATION_STEP_REPORT_PERSISTED = 'report-persisted';

    const GENERATION_STEPS = [
        // Keep the order of this array as this will decide the order of the execution of the generation steps
        self::GENERATION_STEP_REPORT_CREATED,
        self::GENERATION_STEP_REPORT_PREPARED,
        self::GENERATION_STEP_STOCKS_CALCULATED,
        self::GENERATION_STEP_PURCHASES_CALCULATED,
        self::GENERATION_STEP_AVERAGE_PURCHASE_PRICE_CALCULATED,
        self::GENERATION_STEP_STOCK_RATED,
        self::GENERATION_STEP_REPORT_PERSISTED,
    ];
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\GeneratedValue(strategy="IDENTITY")
     * @ORM\Id
     */
    private $id;

    /**
     * @var DateTimeInterface "Stichtag", day of the report
     *
     * @ORM\Column(type="date", nullable=false)
     */
    private $reportingDay;

    /**
     * @var DateTimeInterface All stock entries prior to this time will be used for the report.
     *
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $untilDate = null;

    /**
     * @var bool
     *
     * If a report has generated = false it means it was not generated completely or not at all. (Maybe because of an
     * error during the generation.)
     *
     * @ORM\Column(name="`generated`", type="boolean", nullable=false)
     */
    private $generated = false;

    /**
     * @var string|null
     *
     * @ORM\Column(type="string", nullable=true)
     */
    private $generationStep = null;

    /**
     * @var string|null
     *
     * @ORM\Column(name="comment", type="text", nullable=true)
     */
    private $comment = null;

    /**
     * @var bool
     *
     * @ORM\Column(name="preview", type="boolean")
     */
    private $preview = true;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="ReportRow", mappedBy="report", cascade={"all"})
     */
    private $rows;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     */
    private $method;

    /**
     * @var Warehouse
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse")
     * @ORM\JoinColumn(name="warehouseId", referencedColumnName="id")
     */
    private $warehouse;

    /**
     * @var string
     *
     * @ORM\Column(type="string")
     */
    private $warehouseName;

    /**
     * @var
     *
     * @ORM\Column(type="string")
     */
    private $warehouseCode;

    /**
     * @param DateTimeInterface $reportingDay
     * @param string $method
     * @param Warehouse $warehouse
     */
    public function __construct(DateTimeInterface $reportingDay, $method, Warehouse $warehouse)
    {
        if (!in_array($method, self::METHODS, true)) {
            throw new InvalidArgumentException(
                sprintf('The string "%s" is not a valid method for a stock valuation report', $method)
            );
        }
        $this->setReportingDay($reportingDay);

        $this->generationStep = self::GENERATION_STEPS[0];
        $this->method = $method;
        $this->warehouse = $warehouse;
        $this->warehouseName = $warehouse->getName();
        $this->warehouseCode = $warehouse->getCode();
        $this->rows = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return DateTimeInterface
     */
    public function getReportingDay()
    {
        return $this->reportingDay;
    }

    /**
     * @param DateTimeInterface $reportingDay
     */
    private function setReportingDay(DateTimeInterface $reportingDay)
    {
        // Create a copy of the reporting day in case it is immutable
        $reportingDay = (new DateTime())->setTimestamp($reportingDay->getTimestamp());
        // Remove the time of the reporting day, as it is actually only a date of a day (without time)
        $reportingDay->setTime(0, 0, 0);
        $this->reportingDay = $reportingDay;
    }

    /**
     * @param DateTimeInterface $generationDate Usually you call this with "now" but this is a necessary parameter for
     *        reproducible unit tests.
     */
    public function updateUntilDateBasedOnGenerationDate(DateTimeInterface $generationDate)
    {
        $reportingDay = $this->getReportingDay();
        if ($reportingDay->format('Y-m-d') === $generationDate->format('Y-m-d')) {
            // If the reporting day is the same day as the generation date use the generation date (including the time
            // part!)
            $untilDate = $generationDate;
        } elseif ($reportingDay->getTimestamp() > time()) {
            throw new LogicException(sprintf(
                'The reporting day for report with id=%s could not be updated because the provided day is in the ' .
                'future.',
                $this->getId()
            ));
        } else {
            // If the reporting day is not today, take the stock entries of the whole day into account
            $untilDate = (new DateTime())->setTimestamp(strtotime('+1 day', $reportingDay->getTimestamp()));
        }
        $this->untilDate = $untilDate;
    }

    /**
     * @return string|null
     */
    public function getComment()
    {
        return $this->comment;
    }

    /**
     * @param string|null $comment
     */
    public function setComment($comment)
    {
        $this->comment = $comment;
    }

    /**
     * @return bool
     */
    public function isPreview()
    {
        return $this->preview;
    }

    /**
     * Persists the report by removing the preview flag.
     *
     * @throws StockValuationException
     */
    public function persist()
    {
        if (!$this->isPreview()) {
            // Report is already persisted, nothing to do.
            return;
        }

        if (!$this->isGenerated()) {
            throw new LogicException('The report preview has to be generated before the preview-flag can be removed.');
        }
        if ($this->getReportingDay()->diff($this->getUntilDate())->days === 0) {
            throw StockValuationException::reportDoesNotFullyIncludeReportingDay();
        }
        $this->preview = false;
    }

    /**
     * @return ArrayCollection
     */
    public function getRows()
    {
        return $this->rows;
    }

    /**
     * @return string
     */
    public function getMethod()
    {
        return $this->method;
    }

    /**
     * @return Warehouse
     */
    public function getWarehouse()
    {
        return $this->warehouse;
    }

    /**
     * @return DateTimeInterface|null
     */
    public function getUntilDate()
    {
        return $this->untilDate;
    }

    /**
     * @return bool
     */
    public function isGenerated()
    {
        return $this->generated;
    }

    /**
     * @return string|null
     */
    public function getGenerationStep()
    {
        return $this->generationStep;
    }

    /**
     * @return null|string
     */
    public function getNextGenerationStep()
    {
        $index = array_search($this->generationStep, self::GENERATION_STEPS, true);

        return array_key_exists($index + 1, self::GENERATION_STEPS) ? self::GENERATION_STEPS[$index + 1] : null;
    }

    public function markCurrentGenerationStepAsFinished()
    {
        $this->generationStep = $this->getNextGenerationStep();
        $lastGenerationStep = self::GENERATION_STEPS[count(self::GENERATION_STEPS) - 1];
        if ($this->generationStep === $lastGenerationStep) {
            $this->generated = true;
        }
    }

    /**
     * @return string
     */
    public function getWarehouseName()
    {
        return $this->warehouseName;
    }

    /**
     * @return string
     */
    public function getWarehouseCode()
    {
        return $this->warehouseCode;
    }

    /**
     * @return float
     */
    public function getProgress()
    {
        $index = array_search($this->generationStep, self::GENERATION_STEPS, true);

        return ($index + 1) / count(self::GENERATION_STEPS);
    }
}
