<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\Supplier;

use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\Models\Article\Detail as ArticleDetail;

/**
 * @ORM\Entity
 * @ORM\Table(name="pickware_erp_article_detail_supplier_mappings", uniqueConstraints={
 *      @ORM\UniqueConstraint(name="supplier_articleDetail", columns={
 *          "supplierId",
 *          "articleDetailId"
 *      })
 * })
 */
class ArticleDetailSupplierMapping extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var int
     *
     * @ORM\Column(name="supplierId", type="integer", nullable=false)
     */
    private $supplierId;

    /**
     * @var int
     *
     * @ORM\Column(name="articleDetailId", type="integer", nullable=false, options={"unsigned"=true})
     */
    private $articleDetailId;

    /**
     * @var boolean
     *
     * @ORM\Column(name="defaultSupplier", type="boolean", nullable=false)
     */
    private $defaultSupplier = false;

    /**
     * @var float
     *
     * @ORM\Column(name="purchasePrice", type="decimal", nullable=true, precision=10, scale=2)
     */
    private $purchasePrice;

    /**
     * @var string
     *
     * @ORM\Column(name="supplierArticleNumber", type="string", nullable=true)
     */
    private $supplierArticleNumber;

    /**
     * @var int
     *
     * @ORM\Column(name="minimumOrderAmount", type="integer", nullable=true)
     */
    private $minimumOrderAmount;

    /**
     * @var int
     *
     * @ORM\Column(name="packingUnit", type="integer", nullable=false)
     */
    private $packingUnit = 1;

    /**
     * @var Supplier
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Supplier\Supplier", inversedBy="articleDetailSupplierMappings")
     * @ORM\JoinColumn(name="supplierId", referencedColumnName="id", onDelete="CASCADE")
     */
    private $supplier;

    /**
     * @var ArticleDetail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Article\Detail")
     * @ORM\JoinColumn(name="articleDetailId", referencedColumnName="id", onDelete="CASCADE")
     */
    private $articleDetail;

    /**
     * @param ArticleDetail $articleDetail
     * @param Supplier $supplier
     */
    public function __construct(ArticleDetail $articleDetail, Supplier $supplier)
    {
        $this->articleDetail = $articleDetail;
        $this->supplier = $supplier;
        $this->supplier->getArticleDetailSupplierMappings()->add($this);

        // Explicitly set the both IDs which form the unique identifier of this entity, because Doctrine won't do it,
        // even though we have already set the associated entities
        $this->articleDetailId = $articleDetail->getId();
        $this->supplierId = $supplier->getId();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return boolean
     */
    public function isDefaultSupplier()
    {
        return $this->defaultSupplier;
    }

    /**
     * @param boolean $defaultSupplier
     */
    public function setDefaultSupplier($defaultSupplier)
    {
        $this->defaultSupplier = $defaultSupplier;
    }

    /**
     * @return float
     */
    public function getPurchasePrice()
    {
        return $this->purchasePrice;
    }

    /**
     * @param float $purchasePrice
     */
    public function setPurchasePrice($purchasePrice)
    {
        $this->purchasePrice = $purchasePrice;
    }

    /**
     * @return int
     */
    public function getMinimumOrderAmount()
    {
        return $this->minimumOrderAmount;
    }

    /**
     * @param int $minimumOrderAmount
     */
    public function setMinimumOrderAmount($minimumOrderAmount)
    {
        $this->minimumOrderAmount = $minimumOrderAmount;
    }

    /**
     * @param string $supplierArticleNumber
     */
    public function setSupplierArticleNumber($supplierArticleNumber)
    {
        $this->supplierArticleNumber = $supplierArticleNumber;
    }

    /**
     * @return string
     */
    public function getSupplierArticleNumber()
    {
        return $this->supplierArticleNumber;
    }

    /**
     * @return Supplier
     */
    public function getSupplier()
    {
        return $this->supplier;
    }

    /**
     * @return ArticleDetail
     */
    public function getArticleDetail()
    {
        return $this->articleDetail;
    }

    /**
     * @return int
     */
    public function getPackingUnit()
    {
        return $this->packingUnit;
    }

    /**
     * @param int $packingUnit
     */
    public function setPackingUnit($packingUnit)
    {
        if ($packingUnit <= 0) {
            throw new \InvalidArgumentException(sprintf(
                'Invalid packing unit %d provided, only values greater 0 are allowed.',
                $packingUnit
            ));
        }

        $this->packingUnit = $packingUnit;
    }
}
