<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\SupplierOrder;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\CustomModels\ViisonPickwareERP\Supplier\Supplier;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Shop\Currency;
use Shopware\Models\User\User;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity
 * @ORM\Table(name="pickware_erp_supplier_orders")
 */
class SupplierOrder extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var DateTime
     *
     * @ORM\Column(name="created", type="datetime", nullable=false)
     */
    private $created;

    /**
     * @var string
     *
     * @ORM\Column(name="orderNumber", type="string", length=40, nullable=false)
     */
    private $orderNumber;

    /**
     * @var int
     *
     * @ORM\Column(name="statusId", type="integer", nullable=false)
     */
    private $statusId;

    /**
     * @var float
     *
     * @ORM\Column(name="total", type="decimal", nullable=true, precision=10, scale=2)
     */
    private $total;

    /**
     * @var int
     *
     * @ORM\Column(name="supplierId", type="integer", nullable=true)
     */
    private $supplierId;

    /**
     * @var int
     *
     * @ORM\Column(name="warehouseId", type="integer", nullable=false)
     */
    private $warehouseId;

    /**
     * @var string
     *
     * @ORM\Column(name="comment", type="string", length=255, nullable=true)
     */
    private $comment;

    /**
     * @var string
     *
     * @ORM\Column(name="documentComment", type="string", length=255, nullable=true)
     */
    private $documentComment;

    /**
     * @var DateTime
     *
     * @ORM\Column(name="deliveryDate", type="date", nullable=true)
     */
    private $deliveryDate;

    /**
     * @var DateTime
     *
     * @ORM\Column(name="paymentDueDate", type="date", nullable=true)
     */
    private $paymentDueDate;

    /**
     * @var int
     *
     * @ORM\Column(name="paymentStatus", type="integer", nullable=true)
     */
    private $paymentStatus;

    /**
     * @var int
     *
     * @ORM\Column(name="currencyId", type="integer", nullable=false)
     */
    private $currencyId;

    /**
     * OWNING SIDE
     *
     * @var Supplier
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Supplier\Supplier", inversedBy="supplierOrders")
     * @ORM\JoinColumn(name="supplierId", referencedColumnName="id", onDelete="SET NULL")
     */
    private $supplier;

    /**
     * OWNING SIDE
     *
     * @var Warehouse
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse")
     * @ORM\JoinColumn(name="warehouseId", referencedColumnName="id")
     */
    private $warehouse;

    /**
     * OWNING SIDE
     *
     * @var User
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\User\User")
     * @ORM\JoinColumn(name="userId", referencedColumnName="id")
     */
    private $user;

    /**
     * OWNING SIDE
     *
     * @var SupplierOrderStatus
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderStatus")
     * @ORM\JoinColumn(name="statusId", referencedColumnName="id")
     */
    private $status;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItem", mappedBy="supplierOrder", orphanRemoval=true, cascade={"persist"})
     */
    private $items;

    /**
     * OWNING SIDE
     *
     * @var Currency
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Shop\Currency")
     * @ORM\JoinColumn(name="currencyId", referencedColumnName="id")
     */
    private $currency;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderAttachment", mappedBy="supplierOrder", orphanRemoval=true, cascade={"persist"})
     */
    private $attachments;

    public function __construct()
    {
        $this->items = new ArrayCollection();
        $this->attachments = new ArrayCollection();
        $this->created = new DateTime();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return DateTime
     */
    public function getCreated()
    {
        return $this->created;
    }

    /**
     * @param DateTime|string $created
     */
    public function setCreated($created)
    {
        $this->created = (is_string($created)) ? new DateTime($created) : $created;
    }

    /**
     * @return string
     */
    public function getOrderNumber()
    {
        return $this->orderNumber;
    }

    /**
     * @param string $orderNumber
     */
    public function setOrderNumber($orderNumber)
    {
        $this->orderNumber = $orderNumber;
    }

    /**
     * @return float
     */
    public function getTotal()
    {
        return $this->total;
    }

    /**
     * @param float $total
     */
    public function setTotal($total)
    {
        $this->total = $total;
    }

    /**
     * Recomputes the order total.
     *
     * Computing the order total on the fly using SQL when selecting supplier orders through Doctrine is very slow.
     * Because of this, the order total is cached in a field which needs to be updated every time the items in the
     * supplier order are changed.
     */
    public function recomputeTotal()
    {
        $this->total = 0;

        /** @var SupplierOrderItem $item */
        foreach ($this->getItems() as $item) {
            $this->total = $this->total + $item->getOrderedQuantity() * $item->getPrice();
        }
    }

    /**
     * @return string
     */
    public function getComment()
    {
        return $this->comment;
    }

    /**
     * @param string $comment
     */
    public function setComment($comment)
    {
        $this->comment = $comment;
    }

    /**
     * @return string
     */
    public function getDocumentComment()
    {
        return $this->documentComment;
    }

    /**
     * @param string $documentComment
     */
    public function setDocumentComment($documentComment)
    {
        $this->documentComment = $documentComment;
    }

    /**
     * @return DateTime
     */
    public function getDeliveryDate()
    {
        return $this->deliveryDate;
    }

    /**
     * @param DateTime|string $deliveryDate
     */
    public function setDeliveryDate($deliveryDate)
    {
        $this->deliveryDate = (is_string($deliveryDate)) ? new DateTime($deliveryDate) : $deliveryDate;
    }

    /**
     * @return DateTime
     */
    public function getPaymentDueDate()
    {
        return $this->paymentDueDate;
    }

    /**
     * @param DateTime|string $paymentDueDate
     */
    public function setPaymentDueDate($paymentDueDate)
    {
        $this->paymentDueDate = (is_string($paymentDueDate)) ? new DateTime($paymentDueDate) : $paymentDueDate;
    }

    /**
     * @return int
     */
    public function getPaymentStatus()
    {
        return $this->paymentStatus;
    }

    /**
     * @param int $paymentStatus
     */
    public function setPaymentStatus($paymentStatus)
    {
        $this->paymentStatus = $paymentStatus;
    }

    /**
     * @return Supplier
     */
    public function getSupplier()
    {
        return $this->supplier;
    }

    /**
     *
     * @param Supplier|null $supplier
     */
    public function setSupplier(Supplier $supplier = null)
    {
        $this->supplier = $supplier;
    }

    /**
     * @return Warehouse
     */
    public function getWarehouse()
    {
        return $this->warehouse;
    }

    /**
     * @param Warehouse $warehouse
     */
    public function setWarehouse(Warehouse $warehouse)
    {
        $this->warehouse = $warehouse;
    }

    /**
     * @return User
     */
    public function getUser()
    {
        return $this->user;
    }

    /**
     * @param User $user
     */
    public function setUser(User $user)
    {
        $this->user = $user;
    }

    /**
     * @return SupplierOrderStatus
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * @param SupplierOrderStatus $status
     */
    public function setStatus(SupplierOrderStatus $status)
    {
        $this->status = $status;
    }

    /**
     * @return ArrayCollection
     */
    public function getItems()
    {
        return $this->items;
    }

    /**
     * @return ArrayCollection
     */
    public function getAttachments()
    {
        return $this->attachments;
    }

    /**
     * @return Currency
     */
    public function getCurrency()
    {
        return $this->currency;
    }

    /**
     * @param Currency $currency
     */
    public function setCurrency($currency)
    {
        $this->currency = $currency;
    }

    /**
     * @return int
     */
    public function getCurrencyId()
    {
        return $this->currencyId;
    }

    /**
     * @param int $currencyId
     */
    public function setCurrencyId($currencyId)
    {
        $this->currencyId = $currencyId;
    }
}
