<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\SupplierOrder;

use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\Models\Article\Detail as ArticleDetail;

/**
 * @ORM\Entity
 * @ORM\Table(name="pickware_erp_supplier_order_items")
 */
class SupplierOrderItem extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var int
     *
     * @ORM\Column(name="supplierOrderId", type="integer", nullable=true)
     */
    private $supplierOrderId;

    /**
     * @var int
     *
     * @ORM\Column(name="articleDetailId", type="integer", nullable=true)
     */
    private $articleDetailId;

    /**
     * @var string
     *
     * @ORM\Column(name="name", type="string", length=255, nullable=true)
     */
    private $name;

    /**
     * @var string
     *
     * @ORM\Column(name="orderNumber", type="string", length=40, nullable=false)
     */
    private $orderNumber;

    /**
     * @var int
     *
     * @ORM\Column(name="statusId", type="integer", nullable=false)
     */
    private $statusId;

    /**
     * @var int
     *
     * @ORM\Column(name="orderedQuantity", type="integer")
     */
    private $orderedQuantity;

    /**
     * @var int
     *
     * @ORM\Column(name="deliveredQuantity", type="integer")
     */
    private $deliveredQuantity = 0;

    /**
     * @var float
     *
     * @ORM\Column(name="price", type="decimal", nullable=true, precision=10, scale=2)
     */
    private $price;

    /**
     * @var int
     *
     * @ORM\Column(name="deliveryTime", type="integer", nullable=true)
     */
    private $deliveryTime;

    /**
     * @var string
     *
     * @ORM\Column(name="manufacturerName", type="string", length=255, nullable=true)
     */
    private $manufacturerName;

    /**
     * @var string
     *
     * @ORM\Column(name="manufacturerArticleNumber", type="string", length=255, nullable=true)
     */
    private $manufacturerArticleNumber;

    /**
     * @var string
     *
     * @ORM\Column(name="supplierArticleNumber", type="string", nullable=true)
     */
    private $supplierArticleNumber;

    /**
     * OWNING SIDE
     *
     * @var SupplierOrder
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrder", inversedBy="items")
     * @ORM\JoinColumn(name="supplierOrderId", referencedColumnName="id")
     */
    private $supplierOrder;

    /**
     * @var ArticleDetail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Article\Detail")
     * @ORM\JoinColumn(name="articleDetailId", referencedColumnName="id")
     */
    private $articleDetail;

    /**
     * @var SupplierOrderItemStatus
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItemStatus")
     * @ORM\JoinColumn(name="statusId", referencedColumnName="id")
     */
    private $status;

    /**
     * @param SupplierOrder|null $supplierOrder
     * @param ArticleDetail|null $articleDetail
     */
    public function __construct(SupplierOrder $supplierOrder = null, ArticleDetail $articleDetail = null)
    {
        $this->setSupplierOrder($supplierOrder);
        $this->articleDetail = $articleDetail;
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return SupplierOrder
     */
    public function getSupplierOrder()
    {
        return $this->supplierOrder;
    }

    /**
     * @param SupplierOrder|null $supplierOrder
     */
    public function setSupplierOrder(SupplierOrder $supplierOrder = null)
    {
        if ($this->supplierOrder) {
            $this->supplierOrder->getItems()->removeElement($this);
        }

        $this->supplierOrder = $supplierOrder;

        if ($this->supplierOrder && !$this->supplierOrder->getItems()->contains($this)) {
            $this->supplierOrder->getItems()->add($this);
        }
    }

    /**
     * @return ArticleDetail
     */
    public function getArticleDetail()
    {
        return $this->articleDetail;
    }

    /**
     * @param ArticleDetail|null $articleDetail
     */
    public function setArticleDetail(ArticleDetail $articleDetail = null)
    {
        $this->articleDetail = $articleDetail;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @param string $name
     */
    public function setName($name)
    {
        $this->name = $name;
    }

    /**
     * @return string
     */
    public function getOrderNumber()
    {
        return $this->orderNumber;
    }

    /**
     * @param string $orderNumber
     */
    public function setOrderNumber($orderNumber)
    {
        $this->orderNumber = $orderNumber;
    }

    /**
     * @return SupplierOrderItemStatus
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * @param SupplierOrderItemStatus $status
     */
    public function setStatus(SupplierOrderItemStatus $status)
    {
        $this->status = $status;
    }

    /**
     * @return int
     */
    public function getOrderedQuantity()
    {
        return $this->orderedQuantity;
    }

    /**
     * @param int $orderedQuantity
     */
    public function setOrderedQuantity($orderedQuantity)
    {
        $this->orderedQuantity = $orderedQuantity;
    }

    /**
     * @return int
     */
    public function getDeliveredQuantity()
    {
        return $this->deliveredQuantity;
    }

    /**
     * @param int $deliveredQuantity
     */
    public function setDeliveredQuantity($deliveredQuantity)
    {
        $this->deliveredQuantity = $deliveredQuantity;
    }

    /**
     * @return float
     */
    public function getPrice()
    {
        return $this->price;
    }

    /**
     * @param float $price
     */
    public function setPrice($price)
    {
        $this->price = $price;
    }

    /**
     * @return int
     */
    public function getDeliveryTime()
    {
        return $this->deliveryTime;
    }

    /**
     * @param int $deliveryTime
     */
    public function setDeliveryTime($deliveryTime)
    {
        $this->deliveryTime = $deliveryTime;
    }

    /**
     * @return string
     */
    public function getManufacturerName()
    {
        return $this->manufacturerName;
    }

    /**
     * @param string $manufacturerName
     */
    public function setManufacturerName($manufacturerName)
    {
        $this->manufacturerName = $manufacturerName;
    }

    /**
     * @return string
     */
    public function getManufacturerArticleNumber()
    {
        return $this->manufacturerArticleNumber;
    }

    /**
     * @param string $manufacturerArticleNumber
     */
    public function setManufacturerArticleNumber($manufacturerArticleNumber)
    {
        $this->manufacturerArticleNumber = $manufacturerArticleNumber;
    }

    /**
     * @return string
     */
    public function getSupplierArticleNumber()
    {
        return $this->supplierArticleNumber;
    }

    /**
     * @param string $supplierArticleNumber
     */
    public function setSupplierArticleNumber($supplierArticleNumber)
    {
        $this->supplierArticleNumber = $supplierArticleNumber;
    }
}
