<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\Warehouse;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Shopware\Components\Model\ModelEntity;
use Shopware\Models\Article\Detail as ArticleDetail;

/**
 * @ORM\Entity
 * @ORM\Table(name="pickware_erp_article_detail_bin_location_mappings", uniqueConstraints={
 *      @ORM\UniqueConstraint(name="binLocationId_articleDetailId", columns={
 *          "binLocationId",
 *          "articleDetailId"
 *      })
 * })
 */
class ArticleDetailBinLocationMapping extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var int
     *
     * @ORM\Column(name="binLocationId", type="integer", nullable=false)
     */
    private $binLocationId;

    /**
     * @var int
     *
     * @ORM\Column(name="articleDetailId", type="integer", nullable=false, options={"unsigned"=true})
     */
    private $articleDetailId;

    /**
     * @var int
     *
     * @ORM\Column(name="stock", type="integer", nullable=false)
     */
    private $stock = 0;

    /**
     * @var boolean
     *
     * @ORM\Column(name="defaultMapping", type="boolean", nullable=false)
     */
    private $defaultMapping = false;

    /**
     * @var int
     *
     * @ORM\Column(name="reservedStock", type="integer", nullable=false)
     */
    private $reservedStock = 0;

    /**
     * @var DateTime|null
     *
     * @ORM\Column(name="lastStocktake", type="datetime", nullable=true)
     */
    private $lastStocktake = null;

    /**
     * OWNING SIDE
     *
     * @var BinLocation
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation", inversedBy="articleDetailBinLocationMappings")
     * @ORM\JoinColumn(name="binLocationId", referencedColumnName="id", onDelete="CASCADE")
     */
    private $binLocation;

    /**
     * OWNING SIDE
     *
     * @var ArticleDetail
     *
     * @ORM\ManyToOne(targetEntity="Shopware\Models\Article\Detail")
     * @ORM\JoinColumn(name="articleDetailId", referencedColumnName="id", onDelete="CASCADE")
     */
    private $articleDetail;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\StockLedger\OrderStockReservation", mappedBy="articleDetailBinLocationMapping", cascade={"persist"}))
     */
    private $stockReservations;

    /**
     * @param BinLocation $binLocation
     * @param ArticleDetail $articleDetail
     */
    public function __construct(BinLocation $binLocation, ArticleDetail $articleDetail)
    {
        $this->binLocation = $binLocation;
        $this->articleDetail = $articleDetail;
        $this->stockReservations = new ArrayCollection();

        // Explicitely set the both IDs which form the unique identifier of this entity, because Doctrine
        // won't do it, even though we have already set the assocaited entitis
        $this->binLocationId = $binLocation->getId();
        $this->articleDetailId = $articleDetail->getId();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getStock()
    {
        return $this->stock;
    }

    /**
     * @param int $stock
     */
    public function setStock($stock)
    {
        $this->stock = $stock;
    }

    /**
     * @return boolean
     */
    public function isDefaultMapping()
    {
        return $this->defaultMapping;
    }

    /**
     * @param boolean $defaultMapping
     */
    public function setDefaultMapping($defaultMapping)
    {
        $this->defaultMapping = $defaultMapping;
    }

    /**
     * @return int
     */
    public function getReservedStock()
    {
        return $this->reservedStock;
    }

    /**
     * @param int $reservedStock
     */
    public function setReservedStock($reservedStock)
    {
        $this->reservedStock = $reservedStock;
    }

    /**
     * @return DateTime|null
     */
    public function getLastStocktake()
    {
        return $this->lastStocktake;
    }

    /**
     * @param DateTime|null $lastStocktake
     */
    public function setLastStocktake(DateTime $lastStocktake = null)
    {
        $this->lastStocktake = $lastStocktake;
    }

    /**
     * @return BinLocation
     */
    public function getBinLocation()
    {
        return $this->binLocation;
    }

    /**
     * @param BinLocation $binLocation
     */
    public function setBinLocation(BinLocation $binLocation)
    {
        $this->binLocation = $binLocation;
    }

    /**
     * @return ArticleDetail
     */
    public function getArticleDetail()
    {
        return $this->articleDetail;
    }

    /**
     * @param ArticleDetail $articleDetail
     */
    public function setArticleDetail(ArticleDetail $articleDetail)
    {
        $this->articleDetail = $articleDetail;
    }

    /**
     * @return ArrayCollection
     */
    public function getStockReservations()
    {
        return $this->stockReservations;
    }
}
