<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\Warehouse;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Components\Model\ModelEntity;

/**
 * @ORM\Entity
 * @ORM\Table(name="pickware_erp_bin_locations", uniqueConstraints={
 *      @ORM\UniqueConstraint(name="warehouseId_code", columns={
 *          "warehouseId",
 *          "code"
 *      })
 * })
 */
class BinLocation extends ModelEntity
{
    /**
     * The code used for a warehouse's default bin location.
     */
    const NULL_BIN_LOCATION_CODE = 'pickware_null_bin_location';

    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var int
     *
     * @ORM\Column(name="warehouseId", type="integer", nullable=false)
     */
    private $warehouseId;

    /**
     * @var string
     *
     * @ORM\Column(name="code", type="string", nullable=false)
     */
    private $code;

    /**
     * OWNING SIDE
     *
     * @var Warehouse
     *
     * @ORM\ManyToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse", inversedBy="binLocations")
     * @ORM\JoinColumn(name="warehouseId", referencedColumnName="id", onDelete="CASCADE")
     */
    private $warehouse;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\ArticleDetailBinLocationMapping", mappedBy="binLocation", cascade={"persist"})
     */
    private $articleDetailBinLocationMappings;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\StockLedger\BinLocationStockSnapshot", mappedBy="binLocation", cascade={"persist"}))
     */
    private $stockSnapshots;

    /**
     * @param Warehouse $warehouse
     * @param string $code
     */
    public function __construct(Warehouse $warehouse, $code)
    {
        $this->warehouse = $warehouse;
        $this->code = $code;
        $this->warehouse->getBinLocations()->add($this);
        $this->articleDetailBinLocationMappings = new ArrayCollection();
        $this->stockSnapshots = new ArrayCollection();

        // Explicitly set the the IDs that is part of the unique identifier of this entity, because Doctrine
        // won't do it, even though we have already set the assocaited entity
        $this->warehouseId = $warehouse->getId();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getCode()
    {
        return $this->code;
    }

    /**
     * @param string $code
     */
    public function setCode($code)
    {
        $this->code = $code;
    }

    /**
     * @return Warehouse
     */
    public function getWarehouse()
    {
        return $this->warehouse;
    }

    /**
     * Fetch all ArticleDetailBinLocationMappings.
     *
     * Calling this is potentially very expensive. In particular, NEVER call this on the default BinLocation.
     *
     * @return ArrayCollection
     */
    public function getArticleDetailBinLocationMappings()
    {
        return $this->articleDetailBinLocationMappings;
    }

    /**
     * @return ArrayCollection
     */
    public function getStockSnapshots()
    {
        return $this->stockSnapshots;
    }

    /**
     * @return boolean
     */
    public function isNullBinLocation()
    {
        return $this->equals($this->getWarehouse()->getNullBinLocation());
    }

    /**
     * @param mixed $binLocation
     * @return boolean
     */
    public function equals($binLocation)
    {
        return ($binLocation instanceof BinLocation) && $binLocation->getId() == $this->getId();
    }
}
