<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\Warehouse;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use InvalidArgumentException;
use Shopware\Components\Model\ModelEntity;

/**
 * @ORM\Entity(repositoryClass="WarehouseRepository")
 * @ORM\Table(name="pickware_erp_warehouses", uniqueConstraints={
 *      @ORM\UniqueConstraint(name="code", columns={"code"}),
 *      @ORM\UniqueConstraint(name="name", columns={"name"})
 * })
 */
class Warehouse extends ModelEntity
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer", nullable=false)
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="code", type="string", nullable=false)
     */
    private $code;

    /**
     * @var string
     *
     * @ORM\Column(name="name", type="string", nullable=false)
     */
    private $name;

    /**
     * @var boolean
     *
     * @ORM\Column(name="stockAvailableForSale", type="boolean", nullable=false)
     */
    private $stockAvailableForSale = true;

    /**
     * @var boolean
     *
     * @ORM\Column(name="defaultWarehouse", type="boolean", nullable=false)
     */
    private $defaultWarehouse = false;

    /**
     * @var boolean
     *
     * @ORM\Column(name="defaultReturnShipmentWarehouse", type="boolean", nullable=false)
     */
    private $defaultReturnShipmentWarehouse = false;

    /**
     * @var int
     *
     * @ORM\Column(name="nullBinLocationId", type="integer", nullable=true)
     */
    private $nullBinLocationId;

    /**
     * @var string
     *
     * @ORM\Column(name="contact", type="string", nullable=true)
     */
    private $contact;

    /**
     * @var string
     *
     * @ORM\Column(name="email", type="string", nullable=true)
     */
    private $email;

    /**
     * @var string
     *
     * @ORM\Column(name="phone", type="string", nullable=true)
     */
    private $phone;

    /**
     * @var string
     *
     * @ORM\Column(name="address", type="text", nullable=true)
     */
    private $address;

    /**
     * @var string
     *
     * @ORM\Column(name="comment", type="text", nullable=true)
     */
    private $comment;

    /**
     * @var array
     *
     * @ORM\Column(name="binLocationFormatComponents", type="array", nullable=true)
     */
    private $binLocationFormatComponents;

    /**
     * OWNING SIDE
     *
     * @var BinLocation
     *
     * @ORM\OneToOne(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation", inversedBy="warehouse")
     * @ORM\JoinColumn(name="nullBinLocationId", referencedColumnName="id", onDelete="SET NULL")
     */
    private $nullBinLocation;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\BinLocation", mappedBy="warehouse", cascade={"persist"})
     */
    private $binLocations;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\StockLedger\WarehouseArticleDetailStockCount", mappedBy="warehouse", cascade={"persist"})
     */
    private $articleDetailStockCounts;

    /**
     * INVERSE SIDE
     *
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseArticleDetailConfiguration", mappedBy="warehouse", cascade={"persist"})
     */
    private $articleDetailConfigurations;

    public function __construct()
    {
        $this->binLocations = new ArrayCollection();
        $this->articleDetailStockCounts = new ArrayCollection();
        $this->articleDetailConfigurations = new ArrayCollection();
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getCode()
    {
        return $this->code;
    }

    /**
     * @param string $code
     */
    public function setCode($code)
    {
        $this->code = $code;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @param string $name
     */
    public function setName($name)
    {
        $this->name = $name;
    }

    /**
     * @return boolean
     */
    public function isStockAvailableForSale()
    {
        return $this->stockAvailableForSale;
    }

    /**
     * @param boolean $stockAvailableForSale
     */
    public function setStockAvailableForSale($stockAvailableForSale)
    {
        $this->stockAvailableForSale = $stockAvailableForSale;
    }

    /**
     * @return boolean
     */
    public function isDefaultWarehouse()
    {
        return $this->defaultWarehouse;
    }

    /**
     * @param boolean $defaultWarehouse
     */
    public function setDefaultWarehouse($defaultWarehouse)
    {
        $this->defaultWarehouse = $defaultWarehouse;
    }

    /**
     * @return bool
     */
    public function isDefaultReturnShipmentWarehouse()
    {
        return $this->defaultReturnShipmentWarehouse;
    }

    /**
     * @param bool $defaultReturnShipmentWarehouse
     */
    public function setDefaultReturnShipmentWarehouse($defaultReturnShipmentWarehouse)
    {
        $this->defaultReturnShipmentWarehouse = $defaultReturnShipmentWarehouse;
    }

    /**
     * @return string
     */
    public function getContact()
    {
        return $this->contact;
    }

    /**
     * @param string $contact
     */
    public function setContact($contact)
    {
        $this->contact = $contact;
    }

    /**
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @param string $email
     */
    public function setEmail($email)
    {
        $this->email = $email;
    }

    /**
     * @return string
     */
    public function getPhone()
    {
        return $this->phone;
    }

    /**
     * @param string $phone
     */
    public function setPhone($phone)
    {
        $this->phone = $phone;
    }

    /**
     * @return string
     */
    public function getAddress()
    {
        return $this->address;
    }

    /**
     * @param string $address
     */
    public function setAddress($address)
    {
        $this->address = $address;
    }

    /**
     * @return string
     */
    public function getComment()
    {
        return $this->comment;
    }

    /**
     * @param string $comment
     */
    public function setComment($comment)
    {
        $this->comment = $comment;
    }

    /**
     * @return array
     */
    public function getBinLocationFormatComponents()
    {
        return $this->binLocationFormatComponents;
    }

    /**
     * @param array|null $binLocationFormatComponents
     */
    public function setBinLocationFormatComponents(array $binLocationFormatComponents = null)
    {
        $this->binLocationFormatComponents = $binLocationFormatComponents;
    }

    /**
     * @return BinLocation
     */
    public function getNullBinLocation()
    {
        return $this->nullBinLocation;
    }

    /**
     * @param BinLocation $nullBinLocation
     * @throws InvalidArgumentException if the passed `$nullBinLocation` belongs to a different warehouse than
     *         the receiver.
     */
    public function setNullBinLocation(BinLocation $nullBinLocation)
    {
        if ($nullBinLocation->getWarehouse() !== $this) {
            throw new InvalidArgumentException(sprintf(
                'The passed bin location "%1$s" (ID: %2$d) cannot be set as the null bin location of warehouse "%3$s", because it belongs to a different warehouse "%4$s".',
                $nullBinLocation->getCode(),
                $nullBinLocation->getId(),
                $this->getDisplayName(),
                $nullBinLocation->getWarehouse()->getDisplayName()
            ));
        }
        $this->nullBinLocation = $nullBinLocation;
    }

    /**
     * @return ArrayCollection
     */
    public function getBinLocations()
    {
        return $this->binLocations;
    }

    /**
     * @return ArrayCollection
     */
    public function getArticleDetailStockCounts()
    {
        return $this->articleDetailStockCounts;
    }

    /**
     * @return ArrayCollection
     */
    public function getArticleDetailConfigurations()
    {
        return $this->articleDetailConfigurations;
    }

    /**
     * @return string
     */
    public function getDisplayName()
    {
        return sprintf('%s (%s)', $this->getName(), $this->getCode());
    }

    /**
     * @param mixed $warehouse
     * @return boolean
     */
    public function equals($warehouse)
    {
        return ($warehouse instanceof Warehouse) && $warehouse->getId() == $this->getId();
    }
}
