<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\CustomModels\ViisonPickwareERP\Warehouse;

use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Query;
use Shopware\Components\Model\ModelRepository;
use Shopware\Models\Article\Detail as ArticleDetail;

class WarehouseRepository extends ModelRepository
{
    /**
     * Finds and returns the default warehouse.
     *
     * @return Warehouse
     */
    public function getDefaultWarehouse()
    {
        return $this->findOneBy([
            'defaultWarehouse' => true,
        ]);
    }

    /**
     * Finds and returns the default return shipment warehouse.
     *
     * @return Warehouse
     */
    public function getDefaultReturnShipmentWarehouse()
    {
        return $this->findOneBy([
            'defaultReturnShipmentWarehouse' => true,
        ]);
    }

    /**
     * Returns all BinLocation entities for the given $articleDetail in the given $warehouse. The results are sorted by
     * the mapping's 'defaultMapping' flag (DESC) and the bin location's 'code' (ASC).
     *
     * @param Warehouse $warehouse
     * @param ArticleDetail $articleDetail
     * @return BinLocation[]
     */
    public function findAllBinLocations(Warehouse $warehouse, ArticleDetail $articleDetail)
    {
        return array_map(
            function (ArticleDetailBinLocationMapping $mapping) {
                return $mapping->getBinLocation();
            },
            $this->findSortedWarehouseBinLocationMappings($warehouse, $articleDetail)
        );
    }

    /**
     * Returns all ArticleDetailBinLocationMapping mappings for the given $articleDetail in the given $warehouse.
     * The results are sorted as follows:
     *
     *  1. not default location
     *  2. default mapping
     *  3. code, alphabetically
     *
     * @param Warehouse $warehouse
     * @param ArticleDetail $articleDetail
     * @return ArticleDetailBinLocationMapping[]
     */
    public function findSortedWarehouseBinLocationMappings(Warehouse $warehouse, ArticleDetail $articleDetail)
    {
        $builder = $this->getEntityManager()->createQueryBuilder();
        $builder
            ->select(
                'binLocationMapping',
                'CASE WHEN binLocation = :nullBinLocation THEN 1 ELSE 0 END AS HIDDEN isNullBinLocation'
            )
            ->from(ArticleDetailBinLocationMapping::class, 'binLocationMapping')
            ->join('binLocationMapping.binLocation', 'binLocation')
            ->where('binLocationMapping.articleDetail = :articleDetail')
            ->andWhere('binLocation.warehouse = :warehouse')
            ->orderBy('isNullBinLocation', 'ASC')
            ->addOrderBy('binLocationMapping.defaultMapping', 'DESC')
            ->addOrderBy('binLocation.code', 'ASC')
            ->setParameters([
                'articleDetail' => $articleDetail,
                'nullBinLocation' => $warehouse->getNullBinLocation(),
                'warehouse' => $warehouse,
            ]);

        return $builder->getQuery()->getResult();
    }

    /**
     * Fetches and returns the ArticleDetailBinLocationMapping for a particular ArticleDetail in a
     * particular BinLocation.
     *
     * @param BinLocation $binLocation
     * @param ArticleDetail $articleDetail
     * @param int $hydrationMode
     * @return ArticleDetailBinLocationMapping|null the ArticleDetailBinLocationMapping for the ArticleDetail in the
     *         BinLocation, null if the ArticleDetail does not exist in the BinLocation
     */
    public function findArticleDetailBinLocationMapping(
        ArticleDetail $articleDetail,
        BinLocation $binLocation,
        $hydrationMode = AbstractQuery::HYDRATE_OBJECT
    ) {
        $builder = $this->getEntityManager()->createQueryBuilder();
        $builder
            ->select('binLocationMapping')
            ->from(ArticleDetailBinLocationMapping::class, 'binLocationMapping')
            ->where('binLocationMapping.articleDetail = :articleDetail')
            ->andWhere('binLocationMapping.binLocation = :binLocation')
            ->setParameters([
                'articleDetail' => $articleDetail,
                'binLocation' => $binLocation,
            ]);

        return $builder->getQuery()->getOneOrNullResult($hydrationMode);
    }
}
