<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Backend;

use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;

class BackendAnalyticsSubscriber extends Base
{

    /**
     * The IDs of the shops selected in an analytics request.
     */
    private $selectedShopIds = [];

    /**
     * @return An array containing all subscriptions.
     */
    public static function getSubscribedEvents()
    {
        return [
            'Enlight_Controller_Action_PreDispatch_Backend_Analytics' => 'onPreDispatchBackendAnalytics',
            'Shopware_Analytics_CustomerGroupAmount' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_PartnerRevenue' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_AmountPerCountry' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_AmountPerMonth' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_AmountPerWeek' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_AmountPerWeekday' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_AmountPerHour' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_AmountPerPayment' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_AmountPerShipping' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_ShopStatisticTurnover' => 'onFilterAnalyticsAmountQuery',
            'Shopware_Analytics_ProductAmountPerManufacturer' => 'onFilterAnalyticsProductAmountQuery',
            'Shopware_Analytics_ProductAmountPerCategory' => 'onFilterAnalyticsProductAmountQuery',
            'Shopware_Plugins_ViisonPickwareERP_ViisonPickwareERPAnalyticsMainShopCategory_ProductAmountPerMainShopCategory' => 'onFilterAnalyticsProductAmountQuery',
        ];
    }

    /**
     * Saves the 'selectedShops' request parameter for later use in other hooks.
     *
     * @param args The arguments passed by the method triggering the event.
     */
    public function onPreDispatchBackendAnalytics(\Enlight_Event_EventArgs $args)
    {
        $selectedShops = $args->getSubject()->Request()->getParam('selectedShops');
        if (!empty($selectedShops)) {
            $this->selectedShopIds = explode(',', $selectedShops);
        }
    }

    /**
     * Adds a net turnover result to analytics queries.
     *
     * @param args The arguments passed by the method triggering the event.
     * @return The updated query builder.
     */
    public function onFilterAnalyticsAmountQuery(\Enlight_Event_EventArgs $args)
    {
        $builder = $args->getReturn();

        // Select the total net turnover
        $builder->addSelect('SUM(orders.invoice_amount_net / orders.currencyFactor) AS turnoverNet');

        // Select the net turnover for each selected shop
        foreach ($this->selectedShopIds as $shopId) {
            $shopId = intval($shopId);
            $builder->addSelect(
                'SUM(IF(orders.language=' . $shopId . ', invoice_amount_net/currencyFactor, 0)) as turnoverNet' . $shopId
            );
        }

        return $builder;
    }

    /**
     * Adds a net turnover result to analytics queries.
     *
     * @param args The arguments passed by the method triggering the event.
     * @return The updated query builder.
     */
    public function onFilterAnalyticsProductAmountQuery(\Enlight_Event_EventArgs $args)
    {
        $builder = $args->getReturn();

        // Select the net turnover
        $builder->addSelect(
            'SUM((details.price/(1+IF(orders.net, 0, details.tax_rate)/100) * details.quantity)/currencyFactor) AS turnoverNet'
        );

        return $builder;
    }
}
