<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Backend;

use Doctrine\ORM\EntityManager;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\MultiEdit\Queue;
use Shopware\Models\MultiEdit\QueueArticle;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockLedgerService;
use Shopware_Controllers_Backend_ArticleList;

class BackendArticleListSubscriber extends Base
{
    const DISABLE_STOCK_MANAGEMENT = 'disableStockManagement';
    const ENABLE_STOCK_MANAGEMENT = 'enableStockManagement';

    /**
     * @var array
     */
    private $batchActionArticleDetailsToProcess = [];

    /**
     * @var string|null
     */
    private $batchAction = null;

    /**
     * @inhertidoc
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware_Controllers_Backend_ArticleList::saveSingleEntityAction::before' => 'onBeforeSaveSingleEntityAction',
            'Shopware_Controllers_Backend_ArticleList::getEditableColumnsAction::after' => 'onAfterGetEditableColumnsAction',
            'Shopware_Controllers_Backend_ArticleList::batchAction::before' => 'onBeforeBatchAction',
            'Shopware_Controllers_Backend_ArticleList::batchAction::after' => 'onAfterBatchAction',
        ];
    }

    /**
     * Overwrites the posted 'Detail_inStock' 'Attribute_pickwareIncomingStock' and
     * 'Attribute_pickwarePhysicalStockForSale' values with the respective values currently saved in the database.
     * This is necessary to prevent inconsistencies in the data, because 'inStock' and 'pickwarePhysicalStockForSale'
     * must only be changed by the stock manager and 'pickwareIncomingStock' must only be changed by editing
     * supplier orders.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeSaveSingleEntityAction(\Enlight_Hook_HookArgs $args)
    {
        $request = $args->getSubject()->Request();
        if ($request->getParam('resource') !== 'product') {
            return;
        }

        // Try to find the article detail
        $articleDetailId = $request->getParam('Detail_id', 0);
        $articleDetail = $this->get('models')->find(ArticleDetail::class, $articleDetailId);
        if (!$articleDetail) {
            return;
        }

        // Update overwrite the POSTed inStock, stockMin, pickwareIncomingStock and pickwarePhyiscalStockForSale fields
        // with their current values
        $inStock = $articleDetail->getInStock();
        $request->setPost('Detail_inStock', $inStock);
        $request->setParam('Detail_inStock', $inStock);
        $stockMin = $articleDetail->getStockMin();
        $request->setPost('Detail_stockMin', $stockMin);
        $request->setParam('Detail_stockMin', $stockMin);
        $incomingStock = $articleDetail->getAttribute()->getPickwareIncomingStock();
        $request->setPost('Attribute_pickwareIncomingStock', $incomingStock);
        $request->setParam('Attribute_pickwareIncomingStock', $incomingStock);
        $physicalStockForSale = $articleDetail->getAttribute()->getPickwarePhysicalStockForSale();
        $request->setPost('Attribute_pickwarePhysicalStockForSale', $physicalStockForSale);
        $request->setParam('Attribute_pickwarePhysicalStockForSale', $physicalStockForSale);
    }

    /**
     * Removes the elements named 'Detail.inStock', 'Attribute.pickwareIncomingStock' and
     * 'Attribute.pickwarePhysicalStockForSale' from the responded results to prevent these fields from being edited in
     * the batch processing.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterGetEditableColumnsAction(\Enlight_Hook_HookArgs $args)
    {
        $view = $args->getSubject()->View();
        if (!$view->success || !is_array($view->data)) {
            return;
        }

        // Remove the fields, which should not be editable
        $notEditableFields = [
            'Detail.inStock',
            'Detail.stockMin',
            'Attribute.pickwareIncomingStock',
            'Attribute.pickwarePhysicalStockForSale',
            'Attribute.pickwareStockInitialized',
            'Attribute.pickwareStockInitializationTime',
        ];
        $view->data = array_values(
            array_filter(
                $view->data,
                function ($item) use ($notEditableFields) {
                    return !in_array($item['name'], $notEditableFields);
                }
            )
        );
    }

    /**
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeBatchAction(\Enlight_Hook_HookArgs $args)
    {
        /** @var Shopware_Controllers_Backend_ArticleList $subject */
        $subject = $args->getSubject();
        $queueId = $subject->Request()->getParam('queueId');
        /** @var Queue|null $queue */
        $queue = $this->get('models')->find(Queue::class, $queueId);
        if (!$queue) {
            return;
        }

        // Check if we are modifying the `stockManagementDisabled` attribute of article details.
        $operations = json_decode($queue->getOperations(), true);
        $stockManagementDisabledOperation = $this->getStockManagementDisabledOperation($operations);
        if (!$stockManagementDisabledOperation) {
            return;
        }
        $this->batchAction = $stockManagementDisabledOperation['value'] === '1' ? self::DISABLE_STOCK_MANAGEMENT : self::ENABLE_STOCK_MANAGEMENT;

        // Use Shopwares default of 512 items per batch, same as `BatchProcess::batchProcess()`.
        $numberOfArticleDetailsToProcess = $this->get('config')->getByNamespace('SwagMultiEdit', 'batchItemsPerRequest', 512);
        $queueArticleDetailsToProcess = array_slice($queue->getArticleDetails()->toArray(), 0, $numberOfArticleDetailsToProcess);

        // Save the article details which are to be processed in this request. Only select those which have a
        // `stockManagementDisabled` state that differs from the desired state which will be set by the batch action.
        // This is needed as the queues items will be lost after the `batchAction` finished.
        $this->batchActionArticleDetailsToProcess = array_values(array_filter(array_map(function ($queueArticle) {
            /** @var QueueArticle $queueArticle */
            $articleDetail = $queueArticle->getDetail();

            $attribute = $articleDetail->getAttribute();
            if (!$attribute || $attribute->getPickwareStockManagementDisabled() === $this->batchAction) {
                return null;
            }

            return $articleDetail;
        }, $queueArticleDetailsToProcess)));
    }

    public function onAfterBatchAction()
    {
        if ($this->batchAction === null || count($this->batchActionArticleDetailsToProcess) === 0) {
            return;
        }

        /** @var StockLedgerService $stockLedgerService */
        $stockLedgerService = $this->get('pickware.erp.stock_ledger_service');
        /** @var EntityManager $entityManager */
        $entityManager = $this->get('models');
        /** @var ArticleDetail $articleDetail */
        foreach ($this->batchActionArticleDetailsToProcess as $articleDetail) {
            // Refresh the entities as Shopware's batch action uses plain sql.
            $attribute = $articleDetail->getAttribute();
            $entityManager->refresh($attribute);

            // Verify if the batch action operation was successful.
            $currentAttributeValue = $attribute->getPickwareStockManagementDisabled();
            $expectedAttributeValue = $this->batchAction === self::DISABLE_STOCK_MANAGEMENT;
            if ($currentAttributeValue !== $expectedAttributeValue) {
                // The batch action was not successful, skip this article detail.
                continue;
            }

            // Reset the `stockManagementDisabled` attribute to its previous value as our
            // `stopRecordingStockChangesForArticleDetail` and `startRecordingStockChangesForArticleDetail` methods
            // would not process the article detail.
            $attribute->setPickwareStockManagementDisabled(!$currentAttributeValue);
            $entityManager->flush($attribute);

            if ($this->batchAction === self::DISABLE_STOCK_MANAGEMENT) {
                $stockLedgerService->stopRecordingStockChangesForArticleDetail($articleDetail);
            } else {
                $stockLedgerService->startRecordingStockChangesForArticleDetail($articleDetail);
            }
        }
    }

    /**
     * @param array $operations
     * @return mixed|null
     */
    private function getStockManagementDisabledOperation(array $operations)
    {
        $stockManagementDisabledOperations = array_values(array_filter($operations, function ($operation) {
            return ($operation['column'] === 'Attribute.pickwareStockManagementDisabled');
        }));
        if (count($stockManagementDisabledOperations) === 0) {
            return null;
        }

        return $stockManagementDisabledOperations[0];
    }
}
