<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Backend;

use Shopware\Components\Model\ModelManager;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;

class BackendArticleSubscriber extends Base
{
    const ARTICLE_CREATION_ATTRIBUTE = 'articleCreation';

    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware_Controllers_Backend_Article::getArticleData::after' => 'onAfterGetArticleData',
            'Shopware_Controllers_Backend_Article::detailListAction::after' => 'onAfterDetailListAction',
            'Shopware_Controllers_Backend_Article::saveAction::before' => 'onBeforeSaveAction',
            'Shopware_Controllers_Backend_Article::saveAction::after' => 'onAfterSaveAction',
            'Shopware_Controllers_Backend_Article::saveDetailAction::after' => 'onAfterSaveDetailAction',
            'Shopware_Controllers_Backend_Article::saveArticle::before' => 'onBeforeSaveArticle',
            'Shopware_Controllers_Backend_Article::saveDetail::before' => 'onBeforeSaveDetail',
        ];
    }

    /**
     * Adds the custom article detail fields to the response data.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterGetArticleData(\Enlight_Hook_HookArgs $args)
    {
        $data = $args->getReturn();
        $this->addCustomArticleDetailData($data[0]['mainDetail']);
        $args->setReturn($data);
    }

    /**
     * Adds the custom article detail fields to the response data.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterDetailListAction(\Enlight_Hook_HookArgs $args)
    {
        $data = $args->getSubject()->View()->data;
        if (is_array($data)) {
            foreach ($data as &$detail) {
                $this->addCustomArticleDetailData($detail);
            }
        }
        $args->getSubject()->View()->data = $data;
    }

    /**
     * Checks whether the request contains a 'mainDetail' but no 'id', which indicates the
     * creation of a new article, and, if it does, sets the 'inStock' field to 0 to prevent
     * NULL values in the database.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeSaveAction(\Enlight_Hook_HookArgs $args)
    {
        // Check whether a new 'mainDetail' is being created
        $request = $args->getSubject()->Request();
        $mainDetail = $request->getParam('mainDetail');
        if (!$mainDetail || $request->has('id')) {
            return;
        }

        if ($mainDetail[0]['inStock'] === null) {
            // Set the 'inStock' value to 0 to avoid NULL values in the database
            $mainDetail[0]['inStock'] = 0;
            $request->setParam('mainDetail', $mainDetail);
        }

        // Set a flag in the request to be able to detect the creation later
        $request->setAttribute(self::ARTICLE_CREATION_ATTRIBUTE, true);
    }

    /**
     * Saves the custom article detail fields, in case the main action succeeded,
     * and adds the updated custom fields to the response data.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterSaveAction(\Enlight_Hook_HookArgs $args)
    {
        $view = $args->getSubject()->View();
        if (!$view->success) {
            return;
        }

        if ($args->getSubject()->Request()->getAttribute(self::ARTICLE_CREATION_ATTRIBUTE)) {
            $detailId = $view->data[0]['mainDetail']['id'];
            $articleDetail = $this->get('models')->find(ArticleDetail::class, $detailId);
            if ($articleDetail) {
                $this->get('pickware.erp.stock_initialization_service')->initializeStocksOfArticleDetails([
                    $articleDetail
                ]);
            }
        }

        $responseData = $view->data;
        $requestData = $args->getSubject()->Request()->getParams();
        if (array_key_exists('mainDetail', $requestData) && count($requestData['mainDetail']) >= 1) {
            $this->saveCustomArticleDetailData($responseData[0]['mainDetailId'], $requestData['mainDetail'][0]);
        }

        // Make sure to send the updated custom data
        $responseData = $view->data;
        $this->addCustomArticleDetailData($responseData[0]['mainDetail']);
        $view->data = $responseData;
    }

    /**
     * Saves the custom variant article detail fields, in case the main action succeeded,
     * and adds the updated custom fields to the response data.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterSaveDetailAction(\Enlight_Hook_HookArgs $args)
    {
        $view = $args->getSubject()->View();
        if (!$view->success) {
            return;
        }

        $requestParams = $args->getSubject()->Request()->getParams();
        $this->saveCustomArticleDetailData($requestParams['id'], $requestParams);

        // Make sure to respond the updated custom data
        $responseData = $view->data;
        $this->addCustomArticleDetailData($responseData);
        $view->data = $responseData;
    }

    /**
     * Save the pickwareStockManagementDisabled attribute.
     *
     * @param int $detailId
     * @param array $detail
     */
    protected function saveCustomArticleDetailData($detailId, $detail)
    {
        /** @var ModelManager $modelManager */
        $modelManager = $this->get('models');

        /** @var ArticleDetail $articleDetail */
        $articleDetail = $modelManager->find(ArticleDetail::class, $detailId);
        if (!$articleDetail) {
            return;
        }

        if (isset($detail['pickwareStockManagementDisabled'])) {
            if ($detail['pickwareStockManagementDisabled']) {
                $this->get('pickware.erp.stock_ledger_service')->stopRecordingStockChangesForArticleDetail(
                    $articleDetail
                );
            } else {
                $this->get('pickware.erp.stock_ledger_service')->startRecordingStockChangesForArticleDetail(
                    $articleDetail
                );
            }
        }
    }

    /**
     * @param &array $articleDetailData
     */
    protected function addCustomArticleDetailData(&$articleDetailData)
    {
        // Get the physical stock, that is marked for sale
        $articleDetail = $this->get('models')->find(ArticleDetail::class, $articleDetailData['id']);
        $articleAttribute = $articleDetail->getAttribute();
        if ($articleAttribute !== null) {
            $articleDetailData['pickwarePhysicalStockForSale'] = $articleAttribute->getPickwarePhysicalStockForSale();
            $articleDetailData['pickwareStockManagementDisabled'] = $articleAttribute->getPickwareStockManagementDisabled();
        } else {
            $articleDetailData['pickwarePhysicalStockForSale'] = 0;
            $articleDetailData['pickwareStockManagementDisabled'] = false;
        }

        // Ensure that the (possibly) updated stockMin value is returned
        $articleDetailData['stockMin'] = $articleDetail->getStockMin();
    }

    /**
     * Since the 'inStock' and 'minStock' values are managed by Pickware ERP and saved
     * independently from the rest of an article's data, this hook removes these values
     * from the main detail data, before an article is saved.
     *
     * Please note, that it is fine to remove the 'inStock' and 'stockMin' values in any
     * case even when a new article is created, since it is not possible to edit these
     * values via the Shopware Backend before an article has been saved at least once.
     * This is due to some backend UI manipulations done by the Pickware ERP plugin.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeSaveArticle(\Enlight_Hook_HookArgs $args)
    {
        $data = $args->get('data');

        // Remove 'inStock' and 'stockMin' values if they exist within the provided data
        if (array_key_exists('mainDetail', $data) && count($data['mainDetail']) > 0) {
            if (array_key_exists('inStock', $data['mainDetail'][0])) {
                unset($data['mainDetail'][0]['inStock']);
            }
            if (array_key_exists('stockMin', $data['mainDetail'][0])) {
                unset($data['mainDetail'][0]['stockMin']);
            }
        }

        $args->set('data', $data);
    }

    /**
     * Since the 'inStock' and 'stockMin' values are managed by Pickware ERP and saved
     * independently from the rest of an article detail's data, this hook removes these
     * values from the detail data, before an article detail is saved.
     *
     * Please note, that it is fine to remove the 'inStock' and 'stockMin' values in any
     * case even when a new article detail is created, since it is not possible to edit these
     * values via the Shopware Backend before an article details has been saved at least once.
     * This is due to some backend UI manipulations done by the Pickware ERP plugin.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeSaveDetail(\Enlight_Hook_HookArgs $args)
    {
        $data = $args->get('data');

        // Remove 'inStock' and 'minStock' values if they exist within the provided data
        if (array_key_exists('inStock', $data)) {
            unset($data['inStock']);
        }
        if (array_key_exists('stockMin', $data)) {
            unset($data['stockMin']);
        }

        $args->set('data', $data);
    }
}
