<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Backend;

use DateTime;
use Doctrine\ORM\Query;
use Enlight_Controller_Request_Request as Request;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\ReturnShipment\ReturnShipmentItem;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseRepository;
use Shopware\Models\Attribute\Order as OrderAttribute;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\Order\DetailStatus as OrderDetailStatus;
use Shopware\Models\Order\Document\Document as OrderDocument;
use Shopware\Models\Order\Order;
use Shopware\Models\Order\Repository as OrderRepository;
use Shopware\Models\User\User;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\LocalizedException;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\AbstractBaseSubscriber;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as DocumentUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\OrderDetailUtil;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonCommon\Components\ParameterValidator;
use Shopware\Plugins\ViisonPickwareERP\Components\Cancellation\OrderCanceler;
use Shopware\Plugins\ViisonPickwareERP\Components\Document\CancellationInvoiceContent;
use Shopware\Plugins\ViisonPickwareERP\Components\Document\OrderDocumentCreationService;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityCalculator\OrderDetailQuantityCalculator;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityValidator\OrderDetailQuantityValidator;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockChangeList\StockChangeListFactory;
use Shopware\Plugins\ViisonPickwareERP\Components\StockLedger\StockUpdaterService;
use Shopware_Controllers_Backend_Order;

class BackendOrderSubscriber extends AbstractBaseSubscriber
{
    /**
     * Name of the request attribute that is used to back up the initial quantity values of all items of an order
     * in case that an order PUT action is performed. It is used to track changes to the shipped counter which allows
     * to write corresponding stock entries
     */
    const REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES = 'ViisonPickwareERP_Subscribers_Backend_Order_OriginalShippedValues';

    /**
     * Use a subscriber position that is not 0 because subscribers with position 0 are always executed last.
     * See https://github.com/shopware/shopware/blob/4496e0a25f74d33f681d7b785f3fc3e8f02ecf47/engine/Library/Enlight/Event/EventManager.php#L76
     * and https://github.com/shopware/shopware/blob/4496e0a25f74d33f681d7b785f3fc3e8f02ecf47/engine/Library/Enlight/Event/EventManager.php#L112
     */
    const AFTER_SAVE_ORDER_SUBSCRIBERS_POSITION = 1000;

    /**
     * An array used to back up the initial quantity values of all items of an order
     * in case that an order PUT action is performed.
     *
     * @var array
     */
    private $quantityBackup = [];

    /**
     * We save the user id temporarily here because we cannot access it in onAfterAnySaveAction() anymore (the email
     * generation performed in the save action leads to the session being reset ($shop->registerResources() call in
     * https://github.com/VIISON/shopware/blob/6e801e8be1723694c71175e834d576515f5f572e/engine/Shopware/Core/sOrder.php#L1519).
     *
     * @var int|null
     */
    private $currentUserId = null;

    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware_Controllers_Backend_Order::deleteAction::before' => 'onBeforeDeleteAction',
            'Shopware_Controllers_Backend_Order::deleteAction::after' => 'onAfterDeleteAction',
            'Shopware_Controllers_Backend_Order::getList::after' => 'onAfterGetList',
            'Shopware_Controllers_Backend_Order::getList::before' => 'onBeforeGetList',
            'Shopware_Controllers_Backend_Order::deletePositionAction::after' => 'onAfterDeletePositionAction',
            'Shopware_Controllers_Backend_Order::deletePositionAction::before' => 'onBeforeDeletePositionAction',
            'Shopware_Controllers_Backend_Order::saveAction::before' => [
                ['onBeforeAnySaveAction'],
                ['onBeforeSaveAction'],
            ],
            'Shopware_Controllers_Backend_Order::saveAction::after' => [
                [
                    'onAfterSaveAction',
                    self::AFTER_SAVE_ORDER_SUBSCRIBERS_POSITION,
                ],
            ],
            'Shopware_Controllers_Backend_Order::savePositionAction::before' => [
                ['onBeforeAnySaveAction'],
                ['onBeforeSavePositionAction'],
            ],
            'Shopware_Controllers_Backend_Order::savePositionAction::after' => [
                [
                    'onAfterSavePositionAction',
                    self::AFTER_SAVE_ORDER_SUBSCRIBERS_POSITION,
                ],
            ],
            'Shopware_Controllers_Backend_Order::batchProcessAction::before' => 'onBeforeAnySaveAction',
            'Shopware_Controllers_Backend_Order::batchProcessAction::after' => [
                [
                    'onAfterBatchProcessAction',
                    self::AFTER_SAVE_ORDER_SUBSCRIBERS_POSITION,
                ],
            ],
        ];
    }

    /**
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeGetList(\Enlight_Hook_HookArgs $args)
    {
        $sort = $args->get('sort');

        foreach ($sort as &$sortElement) {
            if ($sortElement['property'] === 'pickwareReturnShipmentStatusId') {
                $sortElement['property'] = 'attribute.' . $sortElement['property'];
            }
        }
        unset($sortElement);

        $args->set('sort', $sort);
    }

    /**
     * Gets all orders contained in the return value and adds custom values such as 'pickwareCanceledQuantity' and
     * 'pickwareIsStockManaged' to each of their positions.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterGetList(\Enlight_Hook_HookArgs $args)
    {
        // Check the result of the original method
        $return = $args->getReturn();
        if (!$return['success']) {
            return;
        }

        // Update the returned order details
        foreach ($return['data'] as &$order) {
            /** @var OrderAttribute $orderAttribute */
            $orderAttribute = $this->get('models')->getRepository(OrderAttribute::class)->findOneBy([
                'orderId' => $order['id'],
            ]);

            // "Monkey patch" the order overwrite protection
            if ($this->shouldMonkeyPatchOrderOverwriteProtection()) {
                $order['pickwareLastChanged'] = ($orderAttribute) ? $orderAttribute->getPickwareLastChanged() : null;
            }

            $order['pickwareReturnShipmentStatusId'] = ($orderAttribute) ? $orderAttribute->getPickwareReturnShipmentStatusId() : null;

            foreach ($order['details'] as &$item) {
                /** @var OrderDetail $orderDetail */
                $orderDetail = $this->get('models')->find(OrderDetail::class, $item['id']);
                $item['pickwareCanceledQuantity'] = ($orderDetail->getAttribute()) ? $orderDetail->getAttribute()->getPickwareCanceledQuantity() : 0;
                /** @var ArticleDetail $articleDetail */
                $articleDetail = $this->get('models')->getRepository(ArticleDetail::class)->findOneBy([
                    'number' => $orderDetail->getArticleNumber(),
                ]);
                $item['pickwareIsStockManaged'] = $articleDetail && $articleDetail->getAttribute() && !$articleDetail->getAttribute()->getPickwareStockManagementDisabled();
            }
            unset($item);
        }
        unset($order);

        $this->addReturnedQuantityToOrderList($return['data']);

        // Update the result
        $args->setReturn($return);
    }

    /**
     * Adds the value 'returnedQuantity' to each order detail of the orders.
     *
     * This is done via one SQL-Query to save execution time.
     *
     * @param array &$orders
     */
    private function addReturnedQuantityToOrderList(array &$orders)
    {
        if (count($orders) === 0) {
            return;
        }

        /** @var \Enlight_Components_Db_Adapter_Pdo_Mysql $db */
        $db = $this->get('db');

        $orderIds = array_column($orders, 'id');
        $returnedQuantities = $db->fetchAssoc(
            'SELECT
                orderDetail.id AS orderDetailId,
                IFNULL(SUM(returnItem.returnedQuantity), 0) AS returnedQuantity,
                IFNULL(SUM(returnItem.cancelledQuantity), 0) AS canceledReturnedQuantity
            FROM s_order AS `order`
            INNER JOIN s_order_details AS orderDetail
                ON orderDetail.orderID = `order`.id
            INNER JOIN pickware_erp_return_shipment_items AS returnItem
                ON returnItem.orderDetailId = orderDetail.id
            WHERE `order`.id IN (' . implode(',', array_fill(0, count($orderIds), '?')) . ')
            GROUP BY orderDetail.id',
            $orderIds
        );

        foreach ($orders as &$order) {
            foreach ($order['details'] as &$orderDetail) {
                $orderDetail['pickwareReturnedQuantity'] = intval($returnedQuantities[$orderDetail['id']]['returnedQuantity'] ?: 0);
                $orderDetail['pickwareCanceledReturnedQuantity'] = intval($returnedQuantities[$orderDetail['id']]['canceledReturnedQuantity'] ?: 0);
            }
        }
    }

    /**
     * Called before an Order is saved via the backend.
     *
     * @param \Enlight_Hook_HookArgs $args The arguments passed by the method triggering the hook.
     * @throws \Exception
     */
    public function onBeforeSaveAction(\Enlight_Hook_HookArgs $args)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var \Shopware_Proxies_ShopwareControllersBackendOrderProxy $subject */
        $subject = $args->getSubject();
        /** @var Order $order */
        $order = $entityManager->find(Order::class, $subject->Request()->getParam('id'));

        // "Monkey patch" the order overwrite protection
        if ($this->shouldMonkeyPatchOrderOverwriteProtection()) {
            // Check whether the order now has a different 'lastUpdated' timestamp than the one provided in the request.
            // That means the order has been changed in the meantime and the changes from the request would reset these
            // changes. Block the saving then.
            $lastChangedTimestamp = new DateTime($args->getSubject()->Request()->getParam('pickwareLastChanged'));
            if ($order && $order->getAttribute() && $lastChangedTimestamp != $order->getAttribute()->getPickwareLastChanged()) {
                throw new \Exception(
                    $this->get('snippets')->getNamespace('backend/viison_pickware_erp_order_protection/main')->get('order_has_been_changed')
                );
            }
        }

        /** @var OrderDetailQuantityValidator $orderDetailQuantityProtector */
        $orderDetailQuantityProtector = $this->get('pickware.erp.order_detail_quantity_validator_service');
        $orderDetailsData = $subject->Request()->getParam('details', []);
        foreach ($orderDetailsData as $orderDetailData) {
            /** @var OrderDetail|null $orderDetail */
            $orderDetail = null;
            if (isset($orderDetailData['id'])) {
                $orderDetail = $entityManager->find(OrderDetail::class, $orderDetailData['id']);
            }

            if ($orderDetail) {
                $newQuantity = ($orderDetailData['quantity'] !== null) ? (int) $orderDetailData['quantity'] : $orderDetail->getQuantity();
                $newShipped = ($orderDetailData['shipped'] !== null) ? (int) $orderDetailData['shipped'] : $orderDetail->getShipped();
                if ($newQuantity !== $orderDetail->getQuantity() || $newShipped !== $orderDetail->getShipped()) {
                    $orderDetailQuantityProtector->validateQuantityAndShippedQuantityCombination($orderDetail, $newQuantity, $newShipped);
                }
            } else {
                $orderDetailQuantityProtector->validateQuantityAndShippedQuantityCombinationForOrderDetailCreation(
                    (int) $orderDetailData['quantity'],
                    (int) $orderDetailData['shipped']
                );
            }
        }

        $originalShippedValues = [];
        // Backs up the shipped and quantity values of all order items to allow the detection of a
        // change in any of the shipped/quantity values.
        /** @var OrderDetail $orderDetail */
        foreach ($order->getDetails() as $orderDetail) {
            $originalShippedValues[$orderDetail->getId()] = $orderDetail->getShipped();
            $this->quantityBackup[$orderDetail->getId()] = $orderDetail->getQuantity();
        }

        $subject->Request()->setAttribute(self::REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES, $originalShippedValues);
    }

    /**
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterSaveAction(\Enlight_Hook_HookArgs $args)
    {
        /** @var Shopware_Controllers_Backend_Order $subject */
        $subject = $args->getSubject();
        if (!$subject->View()->success) {
            return;
        }

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $orderDetailsData = $subject->Request()->getParam('details');
        $orderDetailIds = $orderDetailsData !== null ? array_column($orderDetailsData, 'id') : [];
        /** @var OrderDetail[] $orderDetails */
        $orderDetails = $entityManager->getRepository(OrderDetail::class)->findBy([
            'id' => $orderDetailIds,
        ]);
        foreach ($orderDetails as $orderDetail) {
            $entityManager->refresh($orderDetail);

            $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($orderDetail);

            if (isset($this->quantityBackup[$orderDetail->getId()]) && $articleDetail) {
                // Shopware updates the instock value of an article, which
                // is related to an order position, in a pre-update event of
                // the order detail doctrine model. Apart from that this is
                // bad practice according to the doctrine documentation,
                // the instock value of the loaded article entity is
                // updated, but not persisted along with the order detail.
                // This is due to the fact, that changes made to an entity
                // inside a pre-update event will not be committed in the
                // same flush process. Instead these changes are "pending"
                // until the next flush event.
                // Considering the open nature of the Shopware plugin
                // system it is unpredictable whether or not these changes
                // will be persisted.
                // Furthermore the save action of the backend order
                // controller invokes the entity manager's clear() method,
                // which "kills" the changes of the article entity, whereas
                // the batchProcess action leaves the changes "un-flushed".
                //
                // In order to keep the instock value in sync we need to
                // update the article's instock value manually. Since we
                // perform the update in an after hook, this "manual
                // intervention" will remove the "pending status" by
                // persisting the new instock value.
                //
                // Remark: Please notice, that this approach is not bullet
                // proof: if another plugin registers an after hook as well
                // and if this hook handler is executed before ours and
                // if it invokes a global flush, our approach will (in
                // case of batch processing) lead to doubly updated
                // instock values.
                //
                // TODO Remove this code after fixing the pre-update event
                // handler of the order detail model (pull request).
                //
                // Shopware 5.2.12 (Order Detail Model)
                // https://github.com/shopware/shopware/blob/v5.2.12/engine/Shopware/Models/Order/Detail.php#L607-L640
                //
                // Shopware 5.2.12 (Order Backend Controller - saveAction)
                // https://github.com/shopware/shopware/blob/v5.2.12/engine/Shopware/Controllers/Backend/Order.php#L633-L634
                $quantityDiff = $this->quantityBackup[$orderDetail->getId()] - $orderDetail->getQuantity();
                $articleDetail->setInStock($articleDetail->getInStock() + $quantityDiff);
                $entityManager->flush($articleDetail);
            }
        }

        $warehouse = null;
        $warehouseId = $subject->Request()->getParam('stockChangeWarehouseId', null);
        if ($warehouseId !== null) {
            $warehouse = $entityManager->find(Warehouse::class, $warehouseId);
            ParameterValidator::assertEntityFound($warehouse, Warehouse::class, $warehouseId, 'stockChangeWarehouseId');
        } else {
            /** @var WarehouseRepository $warehouseRepository */
            $warehouseRepository = $entityManager->getRepository(Warehouse::class);
            $warehouse = $warehouseRepository->getDefaultWarehouse();
        }
        $this->writeStockEntriesForShippedQuantityChanges($args, $warehouse);

        /** @var Order $order */
        $order = $entityManager->find(Order::class, $subject->Request()->getParam('id'));

        $this->performRequestedCancellationActions($subject->Request(), $order);
        $pickwareMarkOrderAsShipped = $subject->Request()->getParam('pickwareMarkOrderAsShipped', null);
        if ($pickwareMarkOrderAsShipped) {
            $this->shipShippableOrderDetailsCompletely($order, $warehouse);
        }

        $orderData = array_merge(
            $subject->View()->data,
            $this->getOrderAsArray($order)
        );

        // "Monkey patch" the order overwrite protection:
        // Add the updated "pickwareLastChanged" timestamp to the saveOrder response, so optimistic locking
        // does not block further saves.
        if ($this->shouldMonkeyPatchOrderOverwriteProtection() && $order->getAttribute()) {
            $orderData['pickwareLastChanged'] = $order->getAttribute()->getPickwareLastChanged();
        }

        $subject->View()->assign('data', $orderData);
    }

    /**
     * @param \Enlight_Hook_HookArgs $args The arguments passed by the method triggering the hook.
     * @throws \Exception
     */
    public function onBeforeSavePositionAction(\Enlight_Hook_HookArgs $args)
    {
        // Backs up the shipped and quantity values of the updated order item to allow the detection of
        // a change in those values.
        /** @var OrderDetail $orderDetail */
        $orderDetail = $this->get('models')->find(OrderDetail::class, $args->getSubject()->Request()->getParam('id'));
        if ($orderDetail !== null) {
            $args->getSubject()->Request()->setAttribute(self::REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES, [
                $orderDetail->getId() => $orderDetail->getShipped(),
            ]);
            $this->quantityBackup = [
                $orderDetail->getId() => $orderDetail->getQuantity(),
            ];
        }

        $quantityParam = $args->getSubject()->Request()->getParam('quantity');
        $shippedParam = $args->getSubject()->Request()->getParam('shipped');
        /** @var OrderDetailQuantityValidator $orderDetailQuantityProtector */
        $orderDetailQuantityProtector = $this->get('pickware.erp.order_detail_quantity_validator_service');
        if ($orderDetail) {
            $newQuantity = ($quantityParam !== null) ? (int) $quantityParam : $orderDetail->getQuantity();
            $newShipped = ($shippedParam !== null) ? (int) $shippedParam : $orderDetail->getShipped();
            if ($newQuantity !== $orderDetail->getQuantity() || $newShipped !== $orderDetail->getShipped()) {
                $orderDetailQuantityProtector->validateQuantityAndShippedQuantityCombination($orderDetail, $newQuantity, $newShipped);
            }
        } else {
            $orderDetailQuantityProtector->validateQuantityAndShippedQuantityCombinationForOrderDetailCreation(
                (int) $quantityParam,
                (int) $shippedParam
            );
        }

        // "Monkey patch" the order overwrite protection
        if ($this->shouldMonkeyPatchOrderOverwriteProtection()) {
            // Check whether the order now has a different 'lastUpdated' timestamp than the one provided in the request.
            // That means the order has been changed in the meantime and the changes from the request would reset these
            // changes. Block the saving then.
            /** @var Order $order */
            $order = $this->get('models')->find(Order::class, $args->getSubject()->Request()->getParam('orderId'));
            $lastChangedTimestamp = new DateTime($args->getSubject()->Request()->getParam('pickwareLastChanged'));
            if ($order && $order->getAttribute() && $lastChangedTimestamp != $order->getAttribute()->getPickwareLastChanged()) {
                throw new \Exception(
                    $this->get('snippets')->getNamespace('backend/viison_pickware_erp_order_protection/main')->get('order_has_been_changed')
                );
            }
        }
    }

    /**
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterSavePositionAction(\Enlight_Hook_HookArgs $args)
    {
        $originalShippedValues = $args->getSubject()->Request()->getAttribute(self::REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES);
        $positionData = $args->getSubject()->View()->data;
        if (!isset($originalShippedValues[$positionData['id']])) {
            // Newly created orderDetail, initialize here
            $originalShippedValues[$positionData['id']] = 0;
            $this->quantityBackup[$positionData['id']] = 0;
        }
        $args->getSubject()->Request()->setAttribute(self::REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES, $originalShippedValues);

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        $warehouse = null;
        $warehouseId = $args->getSubject()->Request()->getParam('stockChangeWarehouseId', null);
        if ($warehouseId !== null) {
            $warehouse = $this->get('models')->find(Warehouse::class, $warehouseId);
            ParameterValidator::assertEntityFound($warehouse, Warehouse::class, $warehouseId, 'stockChangeWarehouseId');
        } else {
            /** @var WarehouseRepository $warehouseRepository */
            $warehouseRepository = $entityManager->getRepository(Warehouse::class);
            $warehouse = $warehouseRepository->getDefaultWarehouse();
        }
        $this->writeStockEntriesForShippedQuantityChanges($args, $warehouse);

        // "Monkey patch" the order overwrite protection
        if ($this->shouldMonkeyPatchOrderOverwriteProtection()) {
            $subject = $args->getSubject();

            if (!$subject->View()->success) {
                return;
            }

            $positionData = $subject->View()->data;

            // Add the updated "pickwareLastChanged" timestamp to the savePosition response, so optimistic locking
            // does not block further saves.
            /** @var Order $order */
            $order = $this->get('models')->find(Order::class, $positionData['orderId']);
            if ($order && $order->getAttribute()) {
                $subject->View()->assign('pickwareLastChanged', $order->getAttribute()->getPickwareLastChanged());
            }
        }
    }

    /**
     * Event listener that is called before any save action.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeAnySaveAction(\Enlight_Hook_HookArgs $args)
    {
        // Save the current user for use in onAfterAnySaveAction because the email generation performed in the save
        // action leads to the session being reset ($shop->registerResources() call in
        // https://github.com/VIISON/shopware/blob/6e801e8be1723694c71175e834d576515f5f572e/engine/Shopware/Core/sOrder.php#L1519).
        $this->currentUserId = $this->get('auth')->getIdentity()->id;
    }

    /**
     * Saves the POSTed 'shipped' and 'quantity' values of all details contained in any of the POSTed orders to allow,
     * because the original 'batchProcessAction' does not save detail data.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterBatchProcessAction(\Enlight_Hook_HookArgs $args)
    {
        if (!$args->getSubject()->View()->success) {
            return;
        }

        // Get orders data the same way as in batchProcessAction
        /** @var \Enlight_Controller_Request_Request $request */
        $request = $args->getSubject()->Request();
        $orders = $request->getParam('orders', [
            $request->getParams(),
        ]);

        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        $changedEntities = [];
        $pickwareCancelUnshippedItemsOfOrder = $request->getParam('pickwareCancelUnshippedItemsOfOrder');
        $pickwareMarkOrderAsShipped = $request->getParam('pickwareMarkOrdersAsShipped');
        $warehouse = null;
        if ($pickwareMarkOrderAsShipped) {
            $warehouseId = $request->getParam('stockChangeWarehouseId');
            ParameterValidator::assertIsNotNull($warehouseId, 'stockChangeWarehouseId');
            /** @var Warehouse $warehouse */
            $warehouse = $entityManager->find(Warehouse::class, $warehouseId);
            ParameterValidator::assertEntityFound($warehouse, Warehouse::class, $warehouseId, 'stockChangeWarehouseId');
        }
        foreach ($orders as $orderData) {
            if (!isset($orderData['id'])) {
                continue;
            }
            /** @var Order|null $order */
            $order = $entityManager->find(Order::class, $orderData['id']);
            if (!$order) {
                continue;
            }

            if ($pickwareCancelUnshippedItemsOfOrder) {
                $this->performRequestedCancellationActions($request, $order);
            }
            if ($pickwareMarkOrderAsShipped) {
                $this->shipShippableOrderDetailsCompletely($order, $warehouse);
            }
        }

        // Save changes
        $this->get('models')->flush($changedEntities);
    }

    /**
     * Uses the backed up 'shipped' quantity of the processed order items
     * to add new stock entries, if the 'shipped' value changed.
     *
     * @param \Enlight_Hook_HookArgs $args
     * @param Warehouse $warehouse
     */
    private function writeStockEntriesForShippedQuantityChanges(\Enlight_Hook_HookArgs $args, Warehouse $warehouse)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        // Create a new stock entry based on the changed shipped quantity
        $activeUser = $entityManager->find(User::class, $this->currentUserId);
        /** @var StockChangeListFactory $stockChangeListFactory */
        $stockChangeListFactory = $this->get('pickware.erp.stock_change_list_factory_service');
        /** @var StockUpdaterService $stockUpdater */
        $stockUpdater = $this->get('pickware.erp.stock_updater_service');

        $originalShippedValues = $args->getSubject()->Request()->getAttribute(self::REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES);
        foreach ($originalShippedValues as $orderDetailId => $originalShippedValue) {
            // Try to load the order detail and its article detail
            /** @var OrderDetail|null $orderDetail */
            $orderDetail = $entityManager->find(OrderDetail::class, $orderDetailId);
            if (!$orderDetail) {
                continue;
            }
            $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($orderDetail);
            if (!$articleDetail) {
                continue;
            }

            $totalStockChange = $originalShippedValue - $orderDetail->getShipped();
            if ($totalStockChange !== 0) {
                $stockChanges = $stockChangeListFactory->createStockChangeList(
                    $warehouse,
                    $articleDetail,
                    $totalStockChange
                );
                $stockUpdater->recordOrderDetailShippedChange(
                    $articleDetail,
                    $orderDetail,
                    $stockChanges,
                    $activeUser
                );
            }
        }

        $args->getSubject()->Request()->setAttribute(self::REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES, []);
    }

    /**
     * @var array
     */
    private $deletedOrderDetailsData = [];

    /**
     * Adjust the inStock of all article details associated with the order details of the order that is about ot be
     * deleted to prevent stock errors caused by wrong inStock updates in Shopware's AfterDelete hook.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeDeleteAction(\Enlight_Hook_HookArgs $args)
    {
        $orderId = $args->getSubject()->Request()->getParam('id');
        /** @var Order|null $order */
        $order = $this->get('models')->find(Order::class, $orderId);
        if (!$order) {
            return;
        }

        // Adjust ArticleDetail->inStock so it is increased to the correct value in Shopware's model update hook. That
        // happens in an AfterDelete event of the OrderDetail model. Shopware increases the ArticleDetail->inStock by
        // OrderDetail->quantity. However, the inStock should only increased by number of items left for shipping. So in
        // this hook the ArticleDetail->inStock is reduced by the quantity (to revert SW's subscriber changes) and then
        // increased by the number of items left for shipping.
        $toFlush = [];
        /** @var OrderDetail $orderDetail */
        foreach ($order->getDetails() as $orderDetail) {
            $this->assertOrderDetailRemovalIsAllowed($orderDetail);
            $this->backupOrderDetailDataForDeletion($orderDetail);
        }
        $this->get('models')->flush($toFlush);
    }

    /**
     * @param \Enlight_Hook_HookArgs $args
     * @throws \Exception
     */
    public function onBeforeDeletePositionAction(\Enlight_Hook_HookArgs $args)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var Shopware_Controllers_Backend_Order $subject */
        $subject = $args->getSubject();
        $positions = $subject->Request()->getParam('positions', [['id' => $subject->Request()->getParam('id')]]);

        $orderDetailIds = array_values(array_filter(array_column($positions, 'id')));
        /** @var OrderDetail[] $orderDetails */
        $orderDetails = $entityManager->getRepository(OrderDetail::class)->findBy([
            'id' => $orderDetailIds,
        ]);

        foreach ($orderDetails as $orderDetail) {
            $this->assertOrderDetailRemovalIsAllowed($orderDetail);
            $this->backupOrderDetailDataForDeletion($orderDetail);
        }

        // "Monkey patch" the order overwrite protection
        if ($this->shouldMonkeyPatchOrderOverwriteProtection()) {
            // Check whether the order now has a different 'lastUpdated' timestamp than the one provided in the request.
            // That means the order has been changed in the meantime and the changes from the request would reset these
            // changes. Block the saving then.
            /** @var Order $order */
            $order = $entityManager->find(Order::class, $subject->Request()->getParam('orderID'));
            $lastChangedTimestamp = new DateTime($subject->Request()->getParam('pickwareLastChanged'));
            if ($order && $order->getAttribute() && $lastChangedTimestamp != $order->getAttribute()->getPickwareLastChanged()) {
                throw new \Exception(
                    $this->get('snippets')->getNamespace('backend/viison_pickware_erp_order_protection/main')->get('order_has_been_changed')
                );
            }
        }
    }

    /**
     * @param OrderDetail $orderDetail
     * @throws \Exception
     */
    private function assertOrderDetailRemovalIsAllowed($orderDetail)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        $returnShipmentItemsForOrderDetail = $entityManager->getRepository(ReturnShipmentItem::class)->findBy([
            'orderDetail' => $orderDetail,
        ]);
        if (count($returnShipmentItemsForOrderDetail) !== 0) {
            throw LocalizedException::localize(
                new \Exception(
                    'Order position cannot be removed, because there are still return shipment items referencing ' .
                    'this order item.'
                ),
                'exception/delete_order_detail/return_shipments_exist',
                'viison_pickware_erp/subscribers/backend/backend_order_subscriber'
            );
        }
    }

    /**
     * @param OrderDetail $orderDetail
     */
    private function backupOrderDetailDataForDeletion($orderDetail)
    {
        $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($orderDetail);
        if (!$articleDetail) {
            return;
        }
        /** @var OrderDetailQuantityCalculator $orderDetailQuantityCalculatorService */
        $orderDetailQuantityCalculatorService = $this->get('pickware.erp.order_detail_quantity_calculator_service');
        $this->deletedOrderDetailsData[] = [
            'remainingQuantityToShip' => $orderDetailQuantityCalculatorService->calculateRemainingQuantityToShip($orderDetail),
            'quantity' => $orderDetail->getQuantity(),
            'articleDetail' => $articleDetail,
            'orderDetailId' => $orderDetail->getId(),
        ];
    }

    /**
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterDeleteAction(\Enlight_Hook_HookArgs $args)
    {
        $this->refreshArticleDetailInStockAfterDeletionOfOrderDetails();
    }

    /**
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterDeletePositionAction(\Enlight_Hook_HookArgs $args)
    {
        $this->refreshArticleDetailInStockAfterDeletionOfOrderDetails();

        // "Monkey patch" the order overwrite protection
        if (!$this->shouldMonkeyPatchOrderOverwriteProtection()) {
            return;
        }
        $subject = $args->getSubject();

        if (!$subject->View()->success) {
            return;
        }

        $orderData = $subject->View()->data;

        // Add the updated "pickwareLastChanged" timestamp to the deletePosition response, so optimistic locking
        // does not block further saves of the order.
        /** @var Order $order */
        $order = $this->get('models')->find(Order::class, $orderData['id']);
        if ($order && $order->getAttribute()) {
            $orderData['pickwareLastChanged'] = $order->getAttribute()->getPickwareLastChanged();
        }
        $subject->View()->assign('data', $orderData);
    }

    private function refreshArticleDetailInStockAfterDeletionOfOrderDetails()
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $entitiesToFlush = [];
        foreach ($this->deletedOrderDetailsData as $orderDetailData) {
            if ($orderDetailData['remainingQuantityToShip'] === 0) {
                // No changes necessary
                continue;
            }
            $orderDetail = $entityManager->find(OrderDetail::class, $orderDetailData['orderDetailId']);
            if ($orderDetail) {
                // Order detail was not deleted
                continue;
            }
            /** @var ArticleDetail $articleDetail */
            $articleDetail = $orderDetailData['articleDetail'];
            $oldInStock = $articleDetail->getInStock();
            $articleDetail->setInStock($oldInStock - $orderDetailData['quantity'] + $orderDetailData['remainingQuantityToShip']);
            $entitiesToFlush[] = $articleDetail;
        }
        $entityManager->flush($entitiesToFlush);
    }

    /**
     * @param Request $request
     * @param Order $order
     */
    private function performRequestedCancellationActions(Request $request, Order $order)
    {
        if (!($request->getParam('pickwareCancelUnshippedItemsOfOrder'))) {
            return;
        }

        /** @var OrderCanceler $orderCancelerService */
        $orderCancelerService = $this->get('pickware.erp.order_canceler_service');
        /** @var OrderDetailQuantityCalculator $orderDetailQuantityCalculator */
        $orderDetailQuantityCalculator = $this->get('pickware.erp.order_detail_quantity_calculator_service');
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        $cancellationInvoiceContent = null;
        if ($request->getParam('pickwareCreateCancellationInvoice')) {
            /** @var OrderDocument $invoice */
            $invoice = $order->getDocuments()->filter(function (OrderDocument $orderDocument) {
                return $orderDocument->getTypeId() === DocumentUtil::DOCUMENT_TYPE_ID_INVOICE;
            })->first();
            if ($invoice) {
                $cancellationInvoiceContent = new CancellationInvoiceContent($order, $invoice->getDocumentId());
            }
        }

        if ($request->getParam('pickwareCancelShippingCosts') && $order->getInvoiceShipping() !== 0.0) {
            if ($cancellationInvoiceContent) {
                $cancellationInvoiceContent->setShippingCosts(
                    $order->getInvoiceShipping(),
                    $order->getInvoiceShippingNet(),
                    ViisonCommonUtil::getShippingCostsTaxRateForOrder($order)
                );
            }
            $orderCancelerService->cancelShippingCostsOfOrder($order);
        }
        /** @var OrderDetailStatus $detailStatusOpen */
        $detailStatusOpen = $entityManager->find(
            OrderDetailStatus::class,
            OrderDetailUtil::ORDER_DETAIL_STATUS_ID_OPEN
        );
        /** @var OrderDetail $orderDetail */
        foreach ($order->getDetails() as $orderDetail) {
            $remainingQuantityToShip = $orderDetailQuantityCalculator->calculateRemainingQuantityToShip($orderDetail);
            if ($cancellationInvoiceContent && $remainingQuantityToShip > 0) {
                $cancellationInvoiceContent->addPositionForOrderDetail($orderDetail, $remainingQuantityToShip);
            }

            $orderCancelerService->cancelRemainingQuantityToShipOfOrderDetail($orderDetail, $remainingQuantityToShip);

            $orderDetail->setStatus($detailStatusOpen);
            $entityManager->flush($orderDetail);
        }

        if ($cancellationInvoiceContent && !$cancellationInvoiceContent->isEmpty()) {
            /** @var OrderDocumentCreationService $orderDocumentCreationService */
            $orderDocumentCreationService = $this->get('pickware.erp.order_document_creation_service');
            $cancellationInvoiceContent->createDocument($orderDocumentCreationService, $entityManager);
        }
    }

    /**
     * @param Order $order
     * @param Warehouse $warehouse
     */
    private function shipShippableOrderDetailsCompletely(Order $order, Warehouse $warehouse)
    {
        /** @var OrderDetailQuantityCalculator $orderDetailQuantityCalculator */
        $orderDetailQuantityCalculator = $this->get('pickware.erp.order_detail_quantity_calculator_service');
        /** @var StockChangeListFactory $stockChangeListFactory */
        $stockChangeListFactory = $this->get('pickware.erp.stock_change_list_factory_service');
        /** @var StockUpdaterService $stockUpdater */
        $stockUpdater = $this->get('pickware.erp.stock_updater_service');
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        /** @var User $activeUser */
        $activeUser = $entityManager->find(User::class, $this->currentUserId);
        /** @var OrderDetailStatus $detailStatusCompleted */
        $detailStatusCompleted = $entityManager->find(
            OrderDetailStatus::class,
            OrderDetailUtil::ORDER_DETAIL_STATUS_ID_COMPLETED
        );

        /** @var OrderDetail $orderDetail */
        foreach ($order->getDetails() as $orderDetail) {
            if (!OrderDetailUtil::isOrderDetailForArticleDetail($orderDetail)) {
                continue;
            }

            $leftForShipping = $orderDetailQuantityCalculator->calculateRemainingQuantityToShip($orderDetail);
            $orderDetail->setStatus($detailStatusCompleted);
            $orderDetail->setShipped($orderDetail->getShipped() + $leftForShipping);
            $entityManager->flush($orderDetail);

            $articleDetail = OrderDetailUtil::getArticleDetailForOrderDetail($orderDetail);
            if (!$articleDetail) {
                continue;
            }

            if ($leftForShipping !== 0) {
                $stockChanges = $stockChangeListFactory->createStockChangeList(
                    $warehouse,
                    $articleDetail,
                    -1 * $leftForShipping
                );
                $stockUpdater->recordOrderDetailShippedChange(
                    $articleDetail,
                    $orderDetail,
                    $stockChanges,
                    $activeUser
                );
            }
        }
    }

    /**
     * @param Order $order
     * @return mixed
     */
    private function getOrderAsArray(Order $order)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');
        /** @var OrderRepository $orderRepository */
        $orderRepository = $entityManager->getRepository(Order::class);
        $query = $orderRepository->getOrdersQuery([
            [
                'property' => 'orders.id',
                'value' => $order->getId(),
            ],
        ], []);

        return $query->getOneOrNullResult(Query::HYDRATE_ARRAY);
    }

    /**
     * Checks whether to inject the "order overwrite protection" as "monkey patch".
     *
     * @return bool True as long as the current SW installation does not have the PR
     *         {@see https://github.com/shopware/shopware/pull/1474} merged yet, otherwise false
     */
    private function shouldMonkeyPatchOrderOverwriteProtection()
    {
        return !method_exists(Order::class, 'getChanged');
    }
}
