<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Backend;

use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;

class BackendOverviewSubscriber extends Base
{

    /**
     * @return An array containing all subscriptions.
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware_Controllers_Backend_Overview::getOrderSummaryAction::replace' => 'onReplaceGetOrderSummaryAction',
        ];
    }

    /**
     * Replaces the default 'getOrderSummaryAction' action to add more information to the result.
     *
     * @param args The arguments passed by the method triggering the hook.
     */
    public function onReplaceGetOrderSummaryAction(\Enlight_Hook_HookArgs $args)
    {
        $startDate = $args->getSubject()->Request()->getParam(
            'fromDate',
            date('Y-m-d', mktime(0, 0, 0, date('m'), 1, date('Y')))
        );
        $endDate = $args->getSubject()->Request()->getParam('toDate', date('Y-m-d'));

        $sql = '
            SELECT
                SUM(visitors.uniquevisits) AS visits,
                SUM(visitors.uniquevisits)/SUM(order_count.order_count) AS averageUsers,
                SUM(visitors.pageimpressions) AS hits,
                order_count.order_count AS countOrders,
                customer_count.new_customer_count AS countUsers,
                customer_count.new_customer_order_count AS countCustomers,
                order_amount.amount AS amount,
                order_amount.amountNet AS amountNet,
                visitors.datum AS `date`
            FROM s_statistics_visitors AS visitors
            LEFT OUTER JOIN
            (
                SELECT
                    COUNT(DISTINCT id) AS order_count,
                    DATE (ordertime) AS order_date
                FROM s_order
                WHERE status NOT IN (-1, 4)
                GROUP BY DATE (order_date)
            ) AS order_count
            ON order_count.order_date = visitors.datum
            LEFT OUTER JOIN
            (
                SELECT
                    SUM(invoice_amount/currencyFactor) AS amount,
                    SUM(invoice_amount_net/currencyFactor) AS amountNet,
                    DATE (ordertime) AS order_date
                FROM s_order
                WHERE status NOT IN (-1, 4)
                GROUP BY DATE (order_date)
            ) AS order_amount
            ON order_amount.order_date = visitors.datum
            LEFT OUTER JOIN
            (
                SELECT
                    COUNT(DISTINCT s_user.id) AS new_customer_count,
                    COUNT(DISTINCT s_order.id) AS new_customer_order_count,
                    firstlogin AS first_login_date
                FROM s_user
                LEFT JOIN s_order ON s_order.userID = s_user.id
                    AND (DATE(s_order.ordertime) = DATE(s_user.firstlogin))
                    AND s_order.status NOT IN (-1, 4)
                GROUP BY first_login_date
            ) AS customer_count
            ON customer_count.first_login_date = visitors.datum
            WHERE visitors.datum <= :endDate
                AND visitors.datum >= :startDate
            GROUP BY TO_DAYS(visitors.datum)
            ORDER BY visitors.datum DESC
        ';

        $orders = $this->get('db')->fetchAll(
            $sql,
            [
                'endDate' => $endDate,
                'startDate' => $startDate,
            ]
        );
        foreach ($orders as &$order) {
            // Fix empty values
            foreach ($order as &$value) {
                if (empty($value)) {
                    $value = 0;
                }
            }

            $order['averageOrders'] = (!empty($order['countOrders'])) ? ($order['amount'] / $order['countOrders']) : 0;
            $order['amount'] = round($order['amount'], 2);
            $order['amountNet'] = round($order['amountNet'], 2);
        }

        $args->getSubject()->View()->assign([
            'success' => true,
            'data' => $orders,
            'total' => count($orders),
        ]);
    }
}
