<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Backend;

use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;

class BackendPluginManagerSubscriber extends Base
{
    /**
     * The key used for storing/retrieving the custom pickware category filter in request attributes;
     */
    const REQUEST_ATTR_KEY_PICKWARE_CATEGORY = 'PickwareCategoryFilter';

    /**
     * The whitelist of Pickware core plugins Pickware WMS and Pickware POS.
     */
    const PLUGIN_WHITELIST_CORE_PLUGINS = [
        'ViisonPickwareMobile',
        'ViisonPickwarePOS',
    ];

    /**
     * The whitelist of shipping provider plugins.
     */
    const PLUGIN_WHITELIST_SHIPPING_PROVIDERS = [
        'ViisonAustrianPost',
        'ViisonDHL',
        'ViisonDHLPremium',
        'ViisonDPD',
        'ViisonGermanPost',
        'ViisonGLS',
        'ViisonPostNL',
        'ViisonSwissPost',
        'ViisonUPS',
    ];

    /**
     * The whitelist of payment provider plugins.
     */
    const PLUGIN_WHITELIST_PAYMENT_PROVIDERS = [
        'ViisonBankTransferMatching',
        'ViisonDATEV',
        'ViisonDunning',
        'ViisonOfferManagement',
    ];

    /**
     * The whitelist of processes plugins.
     */
    const PLUGIN_WHITELIST_PROCESSES = [
        'ViisonAddressLabel',
        'ViisonCoupon',
        'ViisonDropShipping',
        'ViisonSetArticles',
    ];

    /**
     * A general blacklist of plugins that should never be listed.
     */
    const PLUGIN_BLACKLIST = [
        'ViisonPickwareERP',
    ];

    /**
     * @override
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware_Controllers_Backend_PluginManager::storeListingAction::before' => 'onBeforeStoreListingAction',
            'Shopware_Controllers_Backend_PluginManager::storeListingAction::after' => 'onAfterStoreListingAction',
        ];
    }

    /**
     * Checks the request for a 'pickwareCategory' filter element and, if found, removes it from the request
     * to prevent errors in the original implementation. Finally the value of the 'pickwareCategory' filter
     * is saved as a request attribute for later use.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeStoreListingAction(\Enlight_Hook_HookArgs $args)
    {
        // Check for a 'pickwareCategory' filter
        $request = $args->getSubject()->Request();
        $filters = $request->getParam('filter', []);
        $pickwareCategoryFilterIndex = -1;
        foreach ($filters as $index => $filter) {
            if ($filter['property'] === 'pickwareCategory') {
                $pickwareCategoryFilterIndex = $index;
                break;
            }
        }
        if ($pickwareCategoryFilterIndex > -1) {
            // Save the custom filter value and remove the element from the request
            $request->setAttribute(self::REQUEST_ATTR_KEY_PICKWARE_CATEGORY, $filters[$pickwareCategoryFilterIndex]['value']);
            unset($filters[$pickwareCategoryFilterIndex]);
            $request->setParam('filter', $filters);
        }
    }

    /**
     * Checks the response for results as well as the request for a Pickware category and, if both are set,
     * applies the category filter on the results. Finally the filtered results are assigned to the view
     * and the number of total results is set to the count of the returned results.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterStoreListingAction(\Enlight_Hook_HookArgs $args)
    {
        $view = $args->getSubject()->View();
        $request = $args->getSubject()->Request();
        if (!$view->success || empty($view->data) || !$request->getAttribute(self::REQUEST_ATTR_KEY_PICKWARE_CATEGORY)) {
            return;
        }

        // Filter the results by the Pickware category
        switch ($request->getAttribute(self::REQUEST_ATTR_KEY_PICKWARE_CATEGORY)) {
            case 'corePlugins':
                $pluginWhitelist = self::PLUGIN_WHITELIST_CORE_PLUGINS;
                break;
            case 'shippingProviders':
                $pluginWhitelist = self::PLUGIN_WHITELIST_SHIPPING_PROVIDERS;
                break;
            case 'paymentProviders':
                $pluginWhitelist = self::PLUGIN_WHITELIST_PAYMENT_PROVIDERS;
                break;
            case 'processes':
                $pluginWhitelist = self::PLUGIN_WHITELIST_PROCESSES;
                break;
            case 'other':
                // Create a whitelist matching all plugins that are not part of any of the other whitelists
                $pluginBlacklist = array_merge(
                    self::PLUGIN_BLACKLIST,
                    self::PLUGIN_WHITELIST_CORE_PLUGINS,
                    self::PLUGIN_WHITELIST_SHIPPING_PROVIDERS,
                    self::PLUGIN_WHITELIST_PAYMENT_PROVIDERS,
                    self::PLUGIN_WHITELIST_PROCESSES
                );
                $pluginWhitelist = array_diff(array_map(function ($plugin) {
                    return $plugin->getTechnicalName();
                }, $view->data), $pluginBlacklist);
                break;
            default:
                return;
        }
        $view->data = array_values(array_filter($view->data, function ($plugin) use ($pluginWhitelist) {
            return in_array($plugin->getTechnicalName(), $pluginWhitelist);
        }));
        $view->total = count($view->data);
    }
}
