<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Components;

use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use Doctrine\ORM\Tools\Pagination\Paginator;
use InvalidArgumentException;
use RuntimeException;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\StockLedgerEntry;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Order\Document\Document as OrderDocument;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as ViisonCommonDocumentUtil;
use Shopware\Plugins\ViisonCommon\Components\ExceptionTranslation\ExceptionTranslator;
use Shopware\Plugins\ViisonCommon\Components\FileStorage\FileStorage;
use Shopware\Plugins\ViisonPickwareERP\Components\Document\PickListDocumentException;
use Shopware_Components_Document;

class DocumentComponentSubscriber extends Base
{
    /**
     * @see \Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base::getSubscribedEvents()
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware_Components_Document::assignValues::after' => 'onAfterAssignValues',
            'Shopware_Components_Document::saveDocument::before' => 'onBeforeSaveDocument',
            'Shopware_Components_Document::render::after' => 'onAfterRender',
        ];
    }

    /**
     * Does one of the following:
     *
     * a) Fixes the document number for cancellation Invoice documents and the referenced invoice number
     * b) Modifies the template variables for pick list documents
     * c) Modifies the template variables for supplier order preview documents (demo data).
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterAssignValues(\Enlight_Hook_HookArgs $args)
    {
        /** @var Shopware_Components_Document $document */
        $document = $args->getSubject();
        $documentTypeId = self::getDocumentTypeId($document);
        $pickListDocumentType = $this->get('pickware.erp.plugin_config_service')->getPickListDocumentType();
        $isPickList = $pickListDocumentType && $documentTypeId === $pickListDocumentType->getId();

        if ($documentTypeId === 4) {
            // Set the cancellation document number and the referenced invoice number again
            $this->setCancellationInvoiceDocumentNumberAndInvoiceNumber($document);
        } elseif ($isPickList) {
            // Assign pick list template variables of the respective order, if this is either a production pick list (no
            // preview), or an order-related pick list preview (orderId is set).
            $this->assignPickListTemplateVariables($document);
        } elseif ($documentTypeId === 5 && $document->_config['_preview']) {
            // Sets demo values for supplier order preview document
            $this->assignSupplierOrderPreviewTemplateVariables($document);
        }
    }

    /**
     * In order to be able to create multiple documents of the same type for one single order,
     * this method sets the user id of the existing order documents to -1. This will influence
     * the hooked saveDocument method in a way, that it will create a new document instead of
     * overwriting the existing one. The modification of these order documents will be reverted
     * after saveDocument has finished, thus in this class' onAfterDocumentSaveDocument method.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforeSaveDocument(\Enlight_Hook_HookArgs $args)
    {
        /** @var Shopware_Components_Document $documentComponent */
        $documentComponent = $args->getSubject();

        // Shopware does specifically not allocate numbers to cancelation of invoice documents. Since Pickware requires
        // these documents to have a number, we take care of allocating a number from the suitable number range.
        $documentTypeId = self::getDocumentTypeId($documentComponent);
        if ($documentTypeId === 4) {
            $documentComponent->_config['invoiceNumber'] = $documentComponent->_config['bid'];

            if (isset($documentComponent->_config['documentId'])) {
                $documentComponent->_config['bid'] = $documentComponent->_config['documentId'];
            } else {
                // Number ranges start at doc_0 for document type 1, so doc_3 is the number range for document type 4
                $documentNumber = $this->get('shopware.number_range_incrementer')->increment('doc_3');
                $documentComponent->_config['bid'] = $documentNumber;
            }
        }
    }

    /**
     * Checks the plugin config for any invoice archive email addresses and, if configured and
     * the document that was rendered is an invoice or cancellation invoice PDF, sends the document
     * file to all archive addresses.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterRender(\Enlight_Hook_HookArgs $args)
    {
        // Check for invoice or cancellation invoice document
        $documentComponent = $args->getSubject();
        $documentTypeId = self::getDocumentTypeId($documentComponent);
        if (!in_array($documentTypeId, [1, 4])) {
            return;
        }

        // Check whether the document was rendered to disk
        $reflectionObject = new \ReflectionObject($documentComponent);
        $rendererProperty = $reflectionObject->getProperty('_renderer');
        $rendererProperty->setAccessible(true);
        $renderer = $rendererProperty->getValue($documentComponent);
        if ($renderer !== 'pdf') {
            return;
        }

        // Check that the document file is readable
        $documentHashProperty = $reflectionObject->getProperty('_documentHash');
        $documentHashProperty->setAccessible(true);
        $documentHash = $documentHashProperty->getValue($documentComponent);

        // Send the document to the invoice archive
        $document = $this->get('models')->getRepository(OrderDocument::class)->findOneBy([
            'hash' => $documentHash,
        ]);
        if (!$document) {
            return;
        }
        $documentFileName = ViisonCommonDocumentUtil::getDocumentFileName($document);
        /** @var FileStorage $documentFileStorage */
        $documentFileStorage = $this->get('viison_common.document_file_storage_service');
        if (!$documentFileStorage->doesFileExist($documentFileName)) {
            return;
        }

        $this->get('pickware.erp.invoice_archive_service')->sendDocumentToInvoiceArchive($document);
    }

    /**
     * @param Shopware_Components_Document $document
     */
    private function setCancellationInvoiceDocumentNumberAndInvoiceNumber(Shopware_Components_Document $document)
    {
        /** @var array $docData */
        $docData = $document->_view->getTemplateVars();
        $docData['Document']['id'] = $document->_config['bid'];
        $docData['Document']['invoiceNumber'] = $document->_config['invoiceNumber'];
        $document->_view->assign('Document', $docData['Document']);
    }

    /**
     * @param Shopware_Components_Document $document
     */
    private function assignPickListTemplateVariables(Shopware_Components_Document $document)
    {
        $orderId = $this->get('front')->Request()->getParam('orderId');
        if (!$document->_config['_preview'] || $orderId) {
            $warehouseId = $this->get('front')->Request()->getParam('pickListWarehouseId');
            if ($warehouseId) {
                $warehouse = $this->get('models')->find(Warehouse::class, $warehouseId);
                if (!$warehouse) {
                    throw new InvalidArgumentException(sprintf('Warehouse with ID %d does not exist.', $warehouseId));
                }
            } else {
                // If the document is created from anywhere but the places we expect (so the warehouse id was not set),
                // we use the default warehouse for the document creation
                $warehouse = $this->get('models')->getRepository(Warehouse::class)->getDefaultWarehouse();
            }

            try {
                $templateVariables = $this->get('pickware.erp.pick_list_document_creation_service')->createPickListDocumentTemplateVariables(
                    $document->_view->getTemplateVars(),
                    $warehouse
                );
                $document->_view->assign($templateVariables);
            } catch (PickListDocumentException $exception) {
                $exceptionTranslator = new ExceptionTranslator($this->get('snippets'));
                throw new RuntimeException($exceptionTranslator->translate($exception));
            }
        } else {
            // Otherwise assign preview pick list template variables (demo data) if this is a general document
            // preview with no real respective order.
            $templateVariables = $this->get('pickware.erp.pick_list_document_creation_service')
                ->createPickListDocumentTemplatePreviewVariables($document);
            $document->_view->assign($templateVariables);
        }
    }

    /**
     * @param Shopware_Components_Document $document
     */
    private function assignSupplierOrderPreviewTemplateVariables(Shopware_Components_Document $document)
    {
        /** @var array $templateVars */
        $templateVars = $document->_view->getTemplateVars();

        $pageItems = [
            [
                'supplierArticleNumber' => 59177,
                'fabricator' => 'Example',
                'articlenumber' => 'SW10177',
                'name' => 'Strandtuch Stripes für Kinder',
                'packUnit' => '1',
                'orderAmount' => 1,
            ],
            [
                'supplierArticleNumber' => 59255,
                'fabricator' => 'Example',
                'articlenumber' => 'SW10210',
                'name' => 'Sonnenmilch',
                'packUnit' => '200ml',
                'orderAmount' => 10,
            ],
            [
                'supplierArticleNumber' => 59192,
                'fabricator' => 'Example',
                'articlenumber' => 'SW10290',
                'name' => 'Strand-Snack Mischung',
                'packUnit' => '100g',
                'orderAmount' => 5,
            ],
            [
                'supplierArticleNumber' => 59243,
                'fabricator' => 'Example',
                'articlenumber' => 'SW10291',
                'name' => 'Eiscreme "Schoko"',
                'packUnit' => '250g',
                'orderAmount' => 3,
            ],
        ];

        $templateVars['Pages'] = [$pageItems];

        $document->_view->assign($templateVars);
    }

    /**
     * Extracts the protected document type of the given document component via reflection.
     *
     * @param Shopware_Components_Document $documentComponent
     * @return int
     */
    private static function getDocumentTypeId(Shopware_Components_Document $documentComponent)
    {
        $reflectionObject = new \ReflectionObject($documentComponent);
        $reflectionProperty = $reflectionObject->getProperty('_typID');
        $reflectionProperty->setAccessible(true);

        return intval($reflectionProperty->getValue($documentComponent));
    }
}
