<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\CoreModules;

use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Document\Document as DocumentType;
use Shopware\Models\Order\Document\Document;
use Shopware\Models\Order\Order;
use Shopware\Plugins\ViisonCommon\Classes\ExceptionHandling\SubscriberExceptionHandling;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;
use Shopware\Plugins\ViisonCommon\Classes\Util\Document as ViisonCommonDocumentUtil;

class OrderModuleSubscriber extends Base
{
    use SubscriberExceptionHandling;

    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware_Modules_Order_SaveOrder_FilterDetailsSQL' => 'onSaveOrderFilterDetailsSQL',
            'Shopware_Modules_Order_SendMail_Send' => 'onSendMailSend',
        ];
    }

    /**
     * Replaces the default '0000-00-00' date in the given INSERT query with the real
     * release date of the associated item or NULL, if the item isn't associated
     * with an article detail or hasn't a release date.
     *
     * @param \Enlight_Event_EventArgs $args
     * @return string
     */
    public function onSaveOrderFilterDetailsSQL(\Enlight_Event_EventArgs $args)
    {
        // Try to find the article detail and determine the new release date
        $basketRow = $args->getRow();
        $articleDetail = $this->get('models')->getRepository(ArticleDetail::class)->findOneByNumber($basketRow['ordernumber']);
        $releaseDateString = 'NULL';
        if ($articleDetail && $articleDetail->getReleaseDate() !== null) {
            $releaseDateString = "'" . $articleDetail->getReleaseDate()->format('Y-m-d') . "'";
        }

        // Replace the default release date in the query and return it
        return str_replace("'0000-00-00',", ($releaseDateString . ','), $args->getReturn());
    }

    /**
     * Checks the mail context for an order and, if it exists and the plugin is configured to
     * created and send invoices via email for the order's dispatch and payment methods,
     * the invoice document is created and attached to the email that is about to be sent to
     * the customer.
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onSendMailSend(\Enlight_Event_EventArgs $args)
    {
        try {
            if (!$this->getPluginConfig()->get('automaticInvoiceCreation')) {
                return;
            }

            // Try to find the order of the email context
            /** @var array $context */
            $context = $args->get('context');
            $orderNumber = $context['sOrderNumber'];
            if (empty($orderNumber)) {
                return;
            }
            /** @var Order $order */
            $order = $this->get('models')->getRepository(Order::class)->findOneBy([
                'number' => $context['sOrderNumber'],
            ]);
            if (!$order) {
                return;
            }

            // Check whether to include the invoice in the email for the given order
            $dispatchWhitelist = $this->getPluginConfig()->toArray()['automaticInvoiceCreationDispatchMethodIds'];
            $paymentWhitelist = $this->getPluginConfig()->toArray()['automaticInvoiceCreationPaymentMethodIds'];
            if ((!empty($dispatchWhitelist) && !in_array($order->getDispatch()->getId(), $dispatchWhitelist))
                || (!empty($paymentWhitelist) && !in_array($order->getPayment()->getId(), $paymentWhitelist))) {
                // Order does not meet whitelist criteria
                return;
            }

            // Create the invoice document
            $invoiceDocumentType = $this->get('models')->find(DocumentType::class, 1);
            /** @var Document $document */
            $document = $this->get('pickware.erp.order_document_creation_service')->createOrderDocument($order, $invoiceDocumentType);

            // Attach invoice to order confirmation mail to customer
            /** @var \Zend_Mail $mail */
            $mail = $args->get('mail');
            ViisonCommonDocumentUtil::attachDocumentToMail($document, $mail);
        } catch (\Exception $e) {
            $this->handleException($e, $args);
        }
    }
}
