<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Doctrine;

use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\Event\OnFlushEventArgs;
use Shopware\Models\Order\Order;
use Shopware\Models\Attribute\Order as OrderAttribute;

/**
 * Doctrine subscriber that uses the Doctrine onFlush event to update the order attribute's pickwareLastChanged property
 * whenever the order is updated.
 *
 * Since we have to update the timestamp whenever Order is updated, but need to make the actual change in Order's
 * Attribute (which is not configured to cascade updates to Order), using preUpdate doesn't work. Also, we cannot use
 * postPersist and postUpdate either, because we cannot safely call $em->flush() in those callbacks.
 *
 * The only viable (and correct) approach to do this is to make the necessary changes in onFlush and then update the
 * unit of work accordingly so the Attribute always gets flushed along with the Order.
 *
 * @see http://docs.doctrine-project.org/projects/doctrine-orm/en/latest/reference/events.html
 */
class DoctrineEventSubscriber implements EventSubscriber
{
    /**
     * @return array
     */
    public function getSubscribedEvents()
    {
        return [
            'onFlush',
        ];
    }

    public function onFlush(OnFlushEventArgs $eventArgs)
    {
        $entityManager = $eventArgs->getEntityManager();
        $unitOfWork = $entityManager->getUnitOfWork();

        // At the moment, Orders are never inserted using Doctrine, but this may change in the future, so we might as
        // well check the insertions too.
        $insertedOrUpdatedEntities = array_merge(
            $unitOfWork->getScheduledEntityInsertions(),
            $unitOfWork->getScheduledEntityUpdates()
        );

        foreach ($insertedOrUpdatedEntities as $entity) {
            if (!($entity instanceof Order)) {
                continue;
            }

            // An order is being inserted or updated, which means we'll have to update the pickwareLastChanged timestamp
            // of that order's attribute.
            /** @var Order $order */
            $order = $entity;
            $orderAttribute = $order->getAttribute();
            $attributeCreated = false;
            if (!$orderAttribute) {
                // This can happen sometimes
                $orderAttribute = new OrderAttribute();
                $order->setAttribute($orderAttribute);
                $attributeCreated = true;
            }

            // Update the "lastChanged" timestamp that stores when the order has been updated the last time.
            $orderAttribute->setPickwareLastChanged(new \DateTime());

            // Update the unit of work's changeset to include the changes we've just made to the attribute.
            $entityManager->persist($orderAttribute);
            $metadata = $entityManager->getClassMetadata(OrderAttribute::class);
            if ($attributeCreated) {
                $unitOfWork->computeChangeSet($metadata, $orderAttribute);
            } else {
                $unitOfWork->recomputeSingleEntityChangeSet($metadata, $orderAttribute);
            }
        }
    }
}
