<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Models\ViisonPickwareERP;

use Shopware\CustomModels\ViisonPickwareERP\SupplierOrder\SupplierOrderItem;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;

class SupplierOrderItemEntityLifecycleSubscriber extends Base
{
    /**
     * @return array
     */
    public static function getSubscribedEvents()
    {
        // Please note, that it is not necessary to subscribe to the postRemove event of supplier orders,
        // since due to the model dependencies (orphan removal) removing a supplier order will internally
        // trigger the removal of all supplier order articles related to that order, which in turn triggers
        // the postRemove event on the supplier order articles.
        return [
            'Shopware\\CustomModels\\ViisonPickwareERP\\SupplierOrder\\SupplierOrder::postUpdate' => 'onSupplierOrderChange',
            'Shopware\\CustomModels\\ViisonPickwareERP\\SupplierOrder\\SupplierOrderItem::postPersist' => 'onSupplierOrderItemChange',
            'Shopware\\CustomModels\\ViisonPickwareERP\\SupplierOrder\\SupplierOrderItem::postUpdate' => 'onSupplierOrderItemChange',
            'Shopware\\CustomModels\\ViisonPickwareERP\\SupplierOrder\\SupplierOrderItem::postRemove' => 'onSupplierOrderItemChange',
        ];
    }

    /**
     * Checks the updated supplier order for a changed 'status' and, if the status has changed,
     * updates the 'pickwareIncomingStock' of all order articles.
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onSupplierOrderChange(\Enlight_Event_EventArgs $args)
    {
        /** @var SupplierOrderItem $orderItem */
        $order = $args->get('entity');
        // Determine whether the 'status' or 'warehouse' has changed and hence the incoming stock of all
        // items must be updated or not. A warehouse change results in invalid cached incoming stocks for the old and
        // new warehouse as the old warehouses cached incoming stocks need to be reduced and the new warehouses cached
        // incoming stocks need to be increased.
        $changeSet = $args->get('entityManager')->getUnitOfWork()->getEntityChangeSet($order);
        if (!isset($changeSet['status']) && !isset($changeSet['warehouse'])) {
            return;
        }

        foreach ($order->getItems() as $orderItem) {
            $this->updateCachedIncomingStock($orderItem);
        }
    }

    /**
     * Updates the 'pickwareIncomingStock' attribute of the article detail that is associated
     * with the supplier order item, that is persisted/updated/removed. When the order item is
     * updated, the incoming stock will only be re-calculated, if the 'orderedQuantity' or
     * 'deliveredQuantity' fields were changed. Furthermore the incoming stock is only updates,
     * if the persists/updated/removed supplier order item is associated with a supplier order.
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onSupplierOrderItemChange(\Enlight_Event_EventArgs $args)
    {
        /** @var SupplierOrderItem $orderItem */
        $orderItem = $args->get('entity');
        // Items must only be considered for the cached incoming stocks, if they are part of a supplier order
        if (!$orderItem->getSupplierOrder()) {
            return;
        }
        if ($args->getName() === 'Shopware\\CustomModels\\ViisonPickwareERP\\SupplierOrder\\SupplierOrderItem::postUpdate') {
            // Determine whether 'supplierOrder', 'orderedQuantity', deliveredQuantity' and/or 'status' have changed
            // and hence the incoming stock must be updated or not
            $changeSet = $args->get('entityManager')->getUnitOfWork()->getEntityChangeSet($orderItem);
            if (count(array_intersect(array_keys($changeSet), ['orderedQuantity', 'deliveredQuantity', 'supplierOrder', 'status'])) === 0) {
                return;
            }
        }

        $this->updateCachedIncomingStock($orderItem);
    }

    /**
     * @param SupplierOrderItem $orderItem
     */
    protected function updateCachedIncomingStock(SupplierOrderItem $orderItem)
    {
        if (!$orderItem->getArticleDetail()) {
            return;
        }
        $derivedPropertyUpdater = $this->get('pickware.erp.derived_property_updater_service');
        $derivedPropertyUpdater->recalculateIncomingStockForArticleDetail(
            $orderItem->getArticleDetail()
        );
    }
}
