<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Models\ViisonPickwareERP;

use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\WarehouseArticleDetailConfiguration;

class WarehouseArticleDetailConfigurationEntityLifecycleSubscriber extends Base
{
    /**
     * @return array
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware\\CustomModels\\ViisonPickwareERP\\Warehouse\\WarehouseArticleDetailConfiguration::postPersist' => 'onPostPersistWarehouseArticleDetailConfiguration',
            'Shopware\\CustomModels\\ViisonPickwareERP\\Warehouse\\WarehouseArticleDetailConfiguration::postUpdate' => 'onPostUpdateWarehouseArticleDetailConfiguration',
            'Shopware\\CustomModels\\ViisonPickwareERP\\Warehouse\\WarehouseArticleDetailConfiguration::postRemove' => 'onPostRemoveWarehouseArticleDetailConfiguration',
        ];
    }

    /**
     * Recalculates the minimum online stock limit ('stockMin') of the related article detail in case that
     * a warehouse article detail configuration entity has been created.
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onPostPersistWarehouseArticleDetailConfiguration(\Enlight_Event_EventArgs $args)
    {
        /** @var WarehouseArticleDetailConfiguration $warehouseStock */
        $warehouseStock = $args->get('entity');

        // Update minimum online stock limit ('stockMin') of related article detail
        $this->get('pickware.erp.derived_property_updater_service')->recalculateMinimumOnlineStockForArticleDetail(
            $warehouseStock->getArticleDetail()
        );
    }

    /**
     * Recalculates the minimum online stock limit of the related article detail in case that the minimum stock value
     * (minimumStock) of an warehouse article detail configuration entity has been changed.
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onPostUpdateWarehouseArticleDetailConfiguration(\Enlight_Event_EventArgs $args)
    {
        /** @var WarehouseArticleDetailConfiguration $warehouseStock */
        $warehouseStock = $args->get('entity');

        // Update minimum stock limit of related article detail if minimum stock has been changed
        $changeSet = $args->get('entityManager')->getUnitOfWork()->getEntityChangeSet($warehouseStock);
        if (isset($changeSet['minimumStock'])) {
            // Update minimum online stock limit ('stockMin') of related article detail
            $this->get('pickware.erp.derived_property_updater_service')->recalculateMinimumOnlineStockForArticleDetail(
                $warehouseStock->getArticleDetail()
            );
        }
    }

    /**
     * Recalculates the minimum online stock limit of the related article detail in case that
     * a warehouse article detail configuration entity has been deleted.
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onPostRemoveWarehouseArticleDetailConfiguration(\Enlight_Event_EventArgs $args)
    {
        /** @var WarehouseArticleDetailConfiguration $warehouseStock */
        $warehouseStock = $args->get('entity');

        // Update minimum online stock limit ('stockMin') of related article detail
        $this->get('pickware.erp.derived_property_updater_service')->recalculateMinimumOnlineStockForArticleDetail(
            $warehouseStock->getArticleDetail()
        );
    }
}
