<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\Models\ViisonPickwareERP;

use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;
use Shopware\CustomModels\ViisonPickwareERP\StockLedger\WarehouseArticleDetailStockCount;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;

class WarehouseEntityLifecycleSubscriber extends Base
{
    /**
     * @return array
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware\\CustomModels\\ViisonPickwareERP\\Warehouse\\Warehouse::postUpdate' => 'onPostUpdateWarehouse',
            'Shopware\\CustomModels\\ViisonPickwareERP\\Warehouse\\Warehouse::postRemove' => 'onPostRemoveWarehouse',
        ];
    }

    /**
     * Checks the updated warehouse for a changed 'stockAvailableForSale' and, if the stockAvailableForSale has changed,
     * recalculates the minimum online stock limit of all related article details.
     *
     * @param \Enlight_Event_EventArgs $args
     */
    public function onPostUpdateWarehouse(\Enlight_Event_EventArgs $args)
    {
        /** @var Warehouse $warehouse */
        $warehouse = $args->get('entity');

        // Update minimum stock limit of all existing article details if the stock avalialbility
        // status of the warehouse has been changed
        $changeSet = $args->get('entityManager')->getUnitOfWork()->getEntityChangeSet($warehouse);
        if (isset($changeSet['stockAvailableForSale'])) {
            // Update minimum online stock limit ('stockMin') of all existing article detaitls
            $this->get('pickware.erp.derived_property_updater_service')->recalculateAllArticleDetailMinimumOnlineStocks();
        }
    }

    /**
     * Recalculates the minimum online stock limit of all related article details in case that
     * a warehouse entity has been deleted.
     */
    public function onPostRemoveWarehouse()
    {
        // Update minimum online stock limit ('stockMin') of all existing article details
        // Please note, that the call of the recalculation method is "save" in the following way:
        // due to the used cascade strategy between warehouse entities and their related
        // warehouse stock entries (cascade is processed directly within the database) the
        // deletion of a warehouse entity via doctrine will NOT trigger the onPostRemoveWarehouseArticleDetailStockCount
        // handler above for each related warehouse stock entity.
        $this->get('pickware.erp.derived_property_updater_service')->recalculateAllArticleDetailMinimumOnlineStocks();
    }
}
