<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\PluginIntegration;

use Enlight_Event_EventArgs;
use Enlight_Hook_HookArgs;
use Shopware\Components\SwagImportExport\DbAdapters;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\Exceptions\Plugins\SwagImportExport\ImportExportException;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\AbstractSwagImportExportIntegrationSubscriber;
use Shopware\Plugins\ViisonPickwareERP\Components\SwagImportExportIntegration\DbAdapters\ViisonPickwareERPArticleStockLimitsDbAdapter;
use Shopware\Plugins\ViisonPickwareERP\Components\SwagImportExportIntegration\DbAdapters\ViisonPickwareERPArticleStocksDbAdapter;
use Shopware\Plugins\ViisonPickwareERP\Components\SwagImportExportIntegration\DbAdapters\ViisonPickwareERPArticleSuppliersDbAdapter;
use Shopware\Plugins\ViisonPickwareERP\Components\SwagImportExportIntegration\DbAdapters\ViisonPickwareERPSuppliersDbAdapter;
use Shopware_Components_Plugin_Bootstrap;

class SwagImportExportIntegrationSubscriber extends AbstractSwagImportExportIntegrationSubscriber
{

    const PROFILE_NAMES = [
        'viison_pickware_erp_article_stocks_absolute' => 'ViisonPickwareERPArticleStocksAbsolute',
        'viison_pickware_erp_article_stocks_relative' => 'ViisonPickwareERPArticleStocksRelative',
        'viison_pickware_erp_article_stock_limits' => 'ViisonPickwareERPArticleStockLimits',
        'viison_pickware_erp_article_suppliers' => 'ViisonPickwareERPArticleSuppliers',
        'viison_pickware_erp_suppliers' => 'ViisonPickwareERPSuppliers',
    ];

    /**
     * @var int[]|null
     */
    protected $uninitializedArticleDetailIds;

    /**
     * @param Shopware_Components_Plugin_Bootstrap $pluginBootstrap
     */
    public function __construct(Shopware_Components_Plugin_Bootstrap $pluginBootstrap)
    {
        parent::__construct(
            $pluginBootstrap,
            self::PROFILE_NAMES,
            'backend/plugins/swag_import_export/',
            $pluginBootstrap->Path() . 'Components/SwagImportExportIntegration/Profiles/'
        );
    }

    /**
     * @inheritdoc
     *
     * Adds more events to the subscription list to add custom functionality of this plugin.
     */
    public static function getSubscribedEvents()
    {
        $parentSubscriptions = parent::getSubscribedEvents();

        return array_merge($parentSubscriptions, [
            // Article initialization
            DbAdapters\ArticlesDbAdapter::class . '::write::before' => 'onBeforeArticlesDbAdapterWrite',
            DbAdapters\ArticlesDbAdapter::class . '::write::after' => 'onAfterArticlesDbAdapterWrite',
            // Article import data manipulation
            'Shopware_Components_SwagImportExport_DbAdapters_ArticlesDbAdapter_Write' => 'onArticlesDbAdapterWrite',
            'Shopware_Components_SwagImportExport_DbAdapters_ArticlesInStockDbAdapter_Write' => 'onArticlesInStockDbAdapterWrite',
        ]);
    }

    /**
     * Saves the IDs of all article details, whose stock has already been initialized.
     *
     * @param Enlight_Hook_HookArgs $args
     */
    public function onBeforeArticlesDbAdapterWrite(Enlight_Hook_HookArgs $args)
    {
        $result = $this->get('pickware.erp.stock_initialization_service')->getAllUninitializedArticleDetails();
        $this->uninitializedArticleDetailIds = array_map(function ($articleDetail) {
            return intval($articleDetail['detailsId']);
        }, $result['data']);
    }

    /**
     * Determines all new article detail's, whose stock has not been initialized, and compares them with the list saved
     * prior to the import. Finally, all uninitialized article details that did not exist before the import are initialized.
     * Also NULL stockMin values for articles that are not managed by the stock manager. This action has to be done after
     * the actual import because you can't import NULL on this field.
     *
     * @param Enlight_Hook_HookArgs $args
     */
    public function onAfterArticlesDbAdapterWrite(Enlight_Hook_HookArgs $args)
    {
        $result = $this->get('pickware.erp.stock_initialization_service')->getAllUninitializedArticleDetails();
        $uninitializedArticleDetailIds = array_map(function ($articleDetail) {
            return intval($articleDetail['detailsId']);
        }, $result['data']);
        $newUninitializedArticleDetailIds = array_diff($uninitializedArticleDetailIds, $this->uninitializedArticleDetailIds);
        if (count($newUninitializedArticleDetailIds) > 0) {
            $articleDetails = $this->getContainer()->get('models')->getRepository(ArticleDetail::class)->findBy([
                'id' => $newUninitializedArticleDetailIds,
            ]);
            $this->get('pickware.erp.stock_initialization_service')->initializeStocksOfArticleDetails($articleDetails);
        }

        // Sets the minimum stock value of articles, which are not stock managed, to NULL
        $records = $args->get('records');
        $importedArticleNumbers = array_column($records['article'], 'orderNumber');
        $this->get('db')->query(
            'UPDATE s_articles_details articleDetail
            LEFT JOIN s_articles_attributes articleAttributes
                ON articleAttributes.articledetailsID = articleDetail.id
            SET
                stockmin = NULL
            WHERE articleAttributes.pickware_stock_management_disabled = 1
                AND ordernumber IN (' . implode(',', array_fill(0, count($importedArticleNumbers), '?')) . ')',
            $importedArticleNumbers
        );
    }

    /**
     * Modifies the import data by overwriting the 'inStock' field of all existing records, whose stock is managed by
     * Pickware, to prevent their 'inStock' from being modified.
     *
     * @param Enlight_Event_EventArgs $args
     * @return mixed
     */
    public function onArticlesDbAdapterWrite(Enlight_Event_EventArgs $args)
    {
        $records = $args->getReturn();
        if (!isset($records['article']) || count($records['article']) === 0) {
            return;
        }

        // Find the numbers and current 'inStock' values of all records that already existing in the database and whose
        // stock is managed by Pickware
        $orderNumbers = array_column($records['article'], 'orderNumber');
        $stockManagedArticleData = $this->get('db')->fetchAll(
            'SELECT ad.ordernumber AS number, IFNULL(ad.instock, 0) AS inStock
            FROM s_articles_details ad
            LEFT JOIN s_articles_attributes aa
                ON aa.articledetailsID = ad.id
            WHERE aa.pickware_stock_management_disabled != 1
            AND ordernumber IN (' . implode(',', array_fill(0, count($orderNumbers), '?')) . ')',
            $orderNumbers
        );
        $stockManagedArticleInStocks = [];
        foreach ($stockManagedArticleData as $articleData) {
            $stockManagedArticleInStocks[$articleData['number']] = $articleData['inStock'];
        }

        if (count($stockManagedArticleInStocks) > 0) {
            // Set the 'inStock' of all existing, stock managed article details to their current 'inStock'
            $stockManagedArticleNumbers = array_keys($stockManagedArticleInStocks);
            foreach ($records['article'] as &$row) {
                if (in_array($row['orderNumber'], $stockManagedArticleNumbers)) {
                    $row['inStock'] = $stockManagedArticleInStocks[$row['orderNumber']];
                }
            }
        }

        return $records;
    }

    /**
     * Modifies the import data by removing all records, whose stock is managed by Pickware, to prevent their 'inStock'
     * from being modified.
     *
     * @param Enlight_Event_EventArgs $args
     * @return mixed
     */
    public function onArticlesInStockDbAdapterWrite(Enlight_Event_EventArgs $args)
    {
        $records = $args->getReturn();
        if (!isset($records['default']) || count($records['default']) === 0) {
            return;
        }

        // Find the numbers of all records that already existing in the database and whose stock is not managed by Pickware
        $orderNumbers = array_column($records['default'], 'orderNumber');
        $notStockManagedArticleNumbers = $this->get('db')->fetchCol(
            'SELECT ad.ordernumber
            FROM s_articles_details ad
            LEFT JOIN s_articles_attributes aa
                ON aa.articledetailsID = ad.id
            WHERE aa.pickware_stock_management_disabled = 1
            AND ordernumber IN (' . implode(',', array_fill(0, count($orderNumbers), '?')) . ')',
            $orderNumbers
        );

        if (count($notStockManagedArticleNumbers) > 0) {
            // Filter the records and leave only the ones whose stock is not managed by Pickware
            $records['default'] = array_values(array_filter(
                $records['default'],
                function ($row) use ($notStockManagedArticleNumbers) {
                    return in_array($row['orderNumber'], $notStockManagedArticleNumbers);
                }
            ));
        } else {
            // Just clear the whole list to speed up the computation
            $records['default'] = [];
        }

        return $records;
    }

    /**
     * @inheritdoc
     */
    public function createDbAdapterOfType($adapterType)
    {
        switch ($adapterType) {
            case 'ViisonPickwareERPArticleStockLimits':
                return new ViisonPickwareERPArticleStockLimitsDbAdapter(
                    $this->get('config'),
                    $this->get('models'),
                    $this->get('snippets')
                );
            case 'ViisonPickwareERPArticleStocksAbsolute':
                return new ViisonPickwareERPArticleStocksDbAdapter(
                    $this->get('config'),
                    $this->get('models'),
                    $this->get('snippets'),
                    $this->get('pickware.erp.stock_ledger_service'),
                    $this->get('pickware.erp.stock_change_list_factory_service'),
                    ViisonPickwareERPArticleStocksDbAdapter::IMPORT_MODE_ABSOLUTE
                );
            case 'ViisonPickwareERPArticleStocksRelative':
                return new ViisonPickwareERPArticleStocksDbAdapter(
                    $this->get('config'),
                    $this->get('models'),
                    $this->get('snippets'),
                    $this->get('pickware.erp.stock_ledger_service'),
                    $this->get('pickware.erp.stock_change_list_factory_service'),
                    ViisonPickwareERPArticleStocksDbAdapter::IMPORT_MODE_RELATIVE
                );
            case 'ViisonPickwareERPArticleSuppliers':
                return new ViisonPickwareERPArticleSuppliersDbAdapter(
                    $this->get('config'),
                    $this->get('models'),
                    $this->get('snippets')
                );
            case 'ViisonPickwareERPSuppliers':
                return new ViisonPickwareERPSuppliersDbAdapter(
                    $this->get('config'),
                    $this->get('models'),
                    $this->get('snippets'),
                    $this->get('pickware.erp.supplier_number_generator_service')
                );
            default:
                // Do nothing if the adapter is not managed by this plugin.
                return null;
        }
    }
}
