<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\RestApi;

use Shopware\Plugins\ViisonCommon\Classes\Subscribers\AbstractBaseSubscriber;
use Shopware\Plugins\ViisonCommon\Classes\Util\Util as ViisonCommonUtil;
use Shopware\Plugins\ViisonPickwareERP\Components\RestApi\RequestLogging\RestApiRequestLoggerService;

class RestApiControllerSubscriber extends AbstractBaseSubscriber
{
    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            'Enlight_Controller_Action_PreDispatch_Api' => 'onPreDispatchApi',
            'Enlight_Controller_Action_PostDispatch_Api' => [
                'onPostDispatchApi',
                1000000,
            ],
            'Enlight_Controller_Front_DispatchLoopStartup' => [
                'onDispatchLoopStartup',
                10,
            ],
        ];
    }

    /**
     * @param \Enlight_Event_EventArgs $args
     */
    public function onPreDispatchApi(\Enlight_Event_EventArgs $args)
    {
        // We need to early return here in case of exceptions already added to the response so they are processed from
        // the shopware error handler. If we do not do that the shopware error handler is not called and no exceptions
        // are returned in the response.
        $exceptions = $args->getResponse()->getException();
        if (count($exceptions) > 0) {
            return;
        }
        $request = $args->getRequest();
        if (!$request->getAttribute(RestApiRequestLoggerService::LOG_RECORD_ID)) {
            $this->get('pickware.erp.rest_api_request_logger_service')->logRequest($request);
        }

        // Furthermore parses the raw JSON post body and sets its content to PHP's global POST variable, to fix the
        // Shopware's broken JSON parsing (This filters null values from the JSON).
        if ($request->getActionName() !== 'invalid' && !empty($request->getRawBody())) {
            // Try to parse the raw body as JSON
            try {
                $json = \Zend_Json::decode($request->getRawBody());
                if (is_array($json) && count($json) > 0) {
                    ViisonCommonUtil::setRequestPostData($json);
                }
            } catch (\Zend_Json_Exception $e) { // phpcs:ignore Generic.CodeAnalysis.EmptyStatement
                // Ignore exception
            }
        }
    }

    /**
     * @param \Enlight_Event_EventArgs $args
     */
    public function onPostDispatchApi(\Enlight_Event_EventArgs $args)
    {
        $this->get('pickware.erp.rest_api_request_logger_service')->logResponse($args->getRequest(), $args->getResponse());
    }

    /**
     * @param \Enlight_Event_EventArgs $args
     */
    public function onDispatchLoopStartup(\Enlight_Event_EventArgs $args)
    {
        // Only modify API request
        $request = $args->getRequest();
        if ($request->getModuleName() !== 'api') {
            return;
        }

        // Applies the custom API routing by changing the request's controller name accordingly.
        $apiRouter = $this->get('pickware.erp.rest_api_router_service');
        $controllerName = $apiRouter->getControllerNameForRequest($request);
        $request->setControllerName($controllerName);
    }
}
