<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\RestApi;

use \Enlight_Controller_Request_Request as Request;
use Shopware\Components\Api\Manager as ResourceManager;
use Shopware\Components\Model\ModelManager;
use Shopware\CustomModels\ViisonPickwareERP\Warehouse\Warehouse;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Models\Order\Detail as OrderDetail;
use Shopware\Models\Order\Order;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\Base;
use Shopware\Plugins\ViisonPickwareERP\Components\OrderDetailQuantityValidator\OrderDetailQuantityValidator;

class RestApiOrdersSubscriber extends Base
{
    const REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES = 'ViisonPickwareERP_Subscribers_Api_Orders_OriginalShippedValues';

    /**
     * Use a subscriber position that is not 0 because subscribers with position 0 are always executed last.
     * See https://github.com/shopware/shopware/blob/4496e0a25f74d33f681d7b785f3fc3e8f02ecf47/engine/Library/Enlight/Event/EventManager.php#L76
     * and https://github.com/shopware/shopware/blob/4496e0a25f74d33f681d7b785f3fc3e8f02ecf47/engine/Library/Enlight/Event/EventManager.php#L112
     */
    const AFTER_PUT_ACTION_SUBSCRIBER_POSITION = 1000;

    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            'Shopware_Controllers_Api_Orders::postAction::after' => 'onAfterPostAction',
            'Shopware_Controllers_Api_Orders::postAction::before' => 'onBeforePostAction',
            'Shopware_Controllers_Api_Orders::putAction::before' => 'onBeforePutAction',
            'Shopware_Controllers_Api_Orders::putAction::after' => [
                'onAfterPutAction',
                self::AFTER_PUT_ACTION_SUBSCRIBER_POSITION,
            ],
        ];
    }

    /**
     * Checks all order details of the created order for positive 'shipped' values and, if they have one, creates a
     * 'sale' stock entry for each of them. If a 'warehouseId' is set in the POST data, the stock entry is created in
     * that warehouse, otherwise the default warehouse is used.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterPostAction(\Enlight_Hook_HookArgs $args)
    {
        $view = $args->getSubject()->View();
        if (!$view->success) {
            return;
        }

        // Try to find the created order
        $order = $this->get('models')->find(Order::class, $view->data['id']);
        if (!$order) {
            return;
        }

        // Find all created order details that have a positive 'shipped' value
        $warehouse = $this->findWarehouse($args->getSubject()->Request());
        $articleDetailRepo = $this->get('models')->getRepository(ArticleDetail::class);
        foreach ($order->getDetails() as $orderDetail) {
            if ($orderDetail->getShipped() <= 0) {
                continue;
            }

            // Shipped set, hence log a 'sale' stock entry, if the order detail has a real article detail
            $articleDetail = $articleDetailRepo->findOneBy([
                'number' => $orderDetail->getArticleNumber(),
            ]);
            if (!$articleDetail) {
                continue;
            }
            $stockChangeList = $this->get('pickware.erp.stock_change_list_factory_service')->createStockChangeList(
                $warehouse,
                $articleDetail,
                -1 * $orderDetail->getShipped()
            );
            $this->get('pickware.erp.stock_ledger_service')->recordSoldStock($articleDetail, $orderDetail, $stockChangeList);
        }
    }

    /**
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforePostAction(\Enlight_Hook_HookArgs $args)
    {
        /** @var OrderDetailQuantityValidator $orderDetailQuantityProtector */
        $orderDetailQuantityProtector = $this->get('pickware.erp.order_detail_quantity_validator_service');

        $orderDetailsData = $args->getSubject()->Request()->getParam('details');
        foreach ($orderDetailsData as $orderDetailData) {
            $orderDetailQuantityProtector->validateQuantityAndShippedQuantityCombinationForOrderDetailCreation(
                (int) $orderDetailData['quantity'],
                (int) $orderDetailData['shipped']
            );
        }
    }

    /**
     * Saves the 'shipped' values of all order details of the order that will be updated by the main action method in
     * a request attribute for later use.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onBeforePutAction(\Enlight_Hook_HookArgs $args)
    {
        /** @var ModelManager $entityManager */
        $entityManager = $this->get('models');

        // Try to find the order that will be updated
        $request = $args->getSubject()->Request();
        $orderId = null;
        $useNumberAsId = boolval($request->getParam('useNumberAsId', 0));
        if ($useNumberAsId) {
            $orderResource = ResourceManager::getResource('order');
            $orderId = $orderResource->getIdFromNumber($request->getParam('id', 0));
        } else {
            $orderId = $request->getParam('id', 0);
        }
        $order = $entityManager->find(Order::class, $orderId);
        if (!$order) {
            return;
        }

        /** @var OrderDetailQuantityValidator $orderDetailQuantityProtector */
        $orderDetailQuantityProtector = $this->get('pickware.erp.order_detail_quantity_validator_service');

        $orderDetailsData = $request->getParam('details');
        foreach ($orderDetailsData as $orderDetailData) {
            /** @var OrderDetail|null $orderDetail */
            $orderDetail = null;
            if (isset($orderDetailData['id'])) {
                $orderDetail = $entityManager->find(OrderDetail::class, $orderDetailData['id']);
            }
            if (!$orderDetail) {
                continue;
            }

            // Api only allows update of 'shipped' value
            $newShipped = ($orderDetailData['shipped'] !== null) ? (int) $orderDetailData['shipped'] : $orderDetail->getShipped();
            if ($newShipped !== $orderDetail->getShipped()) {
                $orderDetailQuantityProtector->validateQuantityAndShippedQuantityCombination($orderDetail, $orderDetail->getQuantity(), $newShipped);
            }
        }

        // Save the current shipped values of all order details as a request attribute
        $shippedValues = [];
        foreach ($order->getDetails() as $orderDetail) {
            $shippedValues[$orderDetail->getId()] = $orderDetail->getShipped();
        }
        $request->setAttribute(self::REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES, $shippedValues);
    }

    /**
     * Checks all order details for a change in their 'shipped' values and, if they have changed, creates a 'sale'
     * (positive change) or 'return' (negative change) stock entry for each of them. If a 'warehouseId' is set in the
     * POST data, the stock entry is created in that warehouse, otherwise the default warehouse is used.
     *
     * @param \Enlight_Hook_HookArgs $args
     */
    public function onAfterPutAction(\Enlight_Hook_HookArgs $args)
    {
        $view = $args->getSubject()->View();
        if (!$view->success) {
            return;
        }

        // Try to find the updated order
        $request = $args->getSubject()->Request();
        $orderId = null;
        $useNumberAsId = boolval($request->getParam('useNumberAsId', 0));
        if ($useNumberAsId) {
            $orderResource = ResourceManager::getResource('order');
            $orderId = $orderResource->getIdFromNumber($request->getParam('id', 0));
        } else {
            $orderId = $request->getParam('id', 0);
        }
        $order = $this->get('models')->find(Order::class, $orderId);
        if (!$order) {
            return;
        }

        // Compare the new 'shipped' values with the original shipped values of all order details
        $warehouse = $this->findWarehouse($request);
        $articleDetailRepo = $this->get('models')->getRepository(ArticleDetail::class);
        $originalShippedValues = $request->getAttribute(self::REQUEST_ATTRIBUTE_ORIGINAL_SHIPPED_VALUES);
        foreach ($order->getDetails() as $orderDetail) {
            $originalShippedValue = (isset($originalShippedValues[$orderDetail->getId()])) ? $originalShippedValues[$orderDetail->getId()] : 0;
            $totalStockChange = $originalShippedValue - $orderDetail->getShipped();
            if ($totalStockChange === 0) {
                continue;
            }

            // Shipped changed, hence log a 'return' or 'sale' stock entry, if the order detail has a real article detail
            $articleDetail = $articleDetailRepo->findOneBy([
                'number' => $orderDetail->getArticleNumber(),
            ]);
            if (!$articleDetail) {
                continue;
            }
            $stockChanges = $this->get('pickware.erp.stock_change_list_factory_service')->createStockChangeList(
                $warehouse,
                $articleDetail,
                $totalStockChange
            );
            $this->get('pickware.erp.stock_updater_service')->recordOrderDetailShippedChange(
                $articleDetail,
                $orderDetail,
                $stockChanges
            );
        }
    }

    /**
     * @param Request $request
     * @return Warehouse
     */
    protected function findWarehouse(Request $request)
    {
        $warehouseId = $request->getParam('warehouseId', 0);
        $warehouseRepo = $this->get('models')->getRepository(Warehouse::class);
        $warehouse = $warehouseRepo->find($warehouseId);
        if (!$warehouse) {
            // Fall back to the default warehouse
            $warehouse = $warehouseRepo->getDefaultWarehouse();
        }

        return $warehouse;
    }
}
