<?php
// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

namespace Shopware\Plugins\ViisonPickwareERP\Subscribers\RestApi;

use Enlight_Event_EventArgs;
use Shopware\Models\Article\Detail as ArticleDetail;
use Shopware\Plugins\ViisonCommon\Classes\Subscribers\AbstractBaseSubscriber;
use Shopware\Plugins\ViisonCommon\Subscribers\Api\RestApiVariantUpdateSubscriber as ViisonCommonVariantUpdateSubscriber;

class RestApiVariantUpdateSubscriber extends AbstractBaseSubscriber
{
    /**
     * @inheritdoc
     */
    public static function getSubscribedEvents()
    {
        return [
            ViisonCommonVariantUpdateSubscriber::FILTER_EVENT_NAME_VARIANT_CREATION_BEFORE => [
                'onBeforeVariantCreation',
                1000, // Use a high position to be able to filter values added by other subscribers
            ],
            ViisonCommonVariantUpdateSubscriber::NOTIFY_EVENT_NAME_VARIANT_CREATION_AFTER => 'onAfterVariantCreation',
            ViisonCommonVariantUpdateSubscriber::FILTER_EVENT_NAME_VARIANT_UPDATE_BEFORE => [
                'onBeforeVariantUpdate',
                1000, // Use a high position to be able to filter values added by other subscribers
            ],
            ViisonCommonVariantUpdateSubscriber::NOTIFY_EVENT_NAME_VARIANT_UPDATE_AFTER => 'onAfterVariantUpdate',
        ];
    }

    /**
     * @param Enlight_Event_EventArgs $args
     * @return array
     */
    public function onBeforeVariantCreation(Enlight_Event_EventArgs $args)
    {
        return $this->get('pickware.erp.rest_api_article_detail_updater_service')->removeCustomAttributesFromPostData(
            $args->getReturn()
        );
    }

    /**
     * @param Enlight_Event_EventArgs $args
     */
    public function onAfterVariantCreation(Enlight_Event_EventArgs $args)
    {
        $variant = $args->get(ViisonCommonVariantUpdateSubscriber::EVENT_ARG_KEY_VARIANT);
        $variantPostData = $args->get(ViisonCommonVariantUpdateSubscriber::EVENT_ARG_KEY_ORIGINAL_VARIANT_POST_DATA);

        $this->get('pickware.erp.stock_initialization_service')->initializeStocksOfArticleDetails([$variant]);

        $restApiArticleDetailUpdater = $this->get('pickware.erp.rest_api_article_detail_updater_service');
        $restApiArticleDetailUpdater->updateStockLimitsWithPostData($variant, $variantPostData);
        $restApiArticleDetailUpdater->updateStockManagementStatusWithPostData($variant, $variantPostData);
    }

    /**
     * @param Enlight_Event_EventArgs $args
     * @return array
     */
    public function onBeforeVariantUpdate(Enlight_Event_EventArgs $args)
    {
        $variantPostData = $args->getReturn();

        $restApiArticleDetailUpdater = $this->get('pickware.erp.rest_api_article_detail_updater_service');
        $variantPostData = $restApiArticleDetailUpdater->removeCustomAttributesFromPostData($variantPostData);
        $variantPostData = $restApiArticleDetailUpdater->removeStockFieldsFromPostData($variantPostData);

        return $variantPostData;
    }

    /**
     * @param Enlight_Event_EventArgs $args
     */
    public function onAfterVariantUpdate(Enlight_Event_EventArgs $args)
    {
        $variant = $args->get(ViisonCommonVariantUpdateSubscriber::EVENT_ARG_KEY_VARIANT);
        $variantPostData = $args->get(ViisonCommonVariantUpdateSubscriber::EVENT_ARG_KEY_ORIGINAL_VARIANT_POST_DATA);

        $restApiArticleDetailUpdater = $this->get('pickware.erp.rest_api_article_detail_updater_service');
        $restApiArticleDetailUpdater->updateStockWithPostData($variant, $variantPostData);
        $restApiArticleDetailUpdater->updateStockManagementStatusWithPostData($variant, $variantPostData);
    }
}
