// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

// By default a grid panel's rendering handles store updates in a performance
// optimized way by only updating rows / cells, which actually needs to be
// updated. Unfortunately a "grouped" grid isn't that smart, meaning a store
// update always triggers a rerendering of the whole list (!). For larger
// grouped list this strategy may lead to a very poor performance, especially
// when the related store is (re-)loaded and the grid is rerendered for each
// loaded / updated record.
//
// This class overwrites the default grouping feature to optimize the list's
// rendering performance in case of store (re-)loads by rerendering the list
// only once.
//
// Please notice, that a grid is automatically rerenderd after its related
// store has finished a (re-)load. Hence there is no need to handle the
// "final" rerendering manually.
Ext.define('ViisonPickwareERP.Ext.grid.feature.Grouping', {

    extend: 'Ext.grid.feature.Grouping',
    alias: 'feature.pickware-erp-grouping',

    // Flag indicating whether or not the related store is in loading mode
    storeIsLoading: false,

    // Ensures that the loading flag is always in sync with the actual
    // state of the store by register suitable event listeners
    init: function () {
        if (this.view.store.isLoading()) {
            this.storeIsLoading = true;
        }

        this.view.store.on('beforeload', function () {
            this.storeIsLoading = true;
        }, this);

        this.view.store.on('load', function () {
            this.storeIsLoading = false;
        }, this);

        this.view.on('refresh', function () {
            this.viisonPickwareErpFixIndexes();
        }, this);

        this.callParent(arguments);
    },

    // Prevents view updates while the store is loading
    onUpdate: function () {
        if (this.storeIsLoading) {
            return;
        }

        this.callParent(arguments);
    },

    /**
     * The view's default binding of dom nodes and records does not
     * consider record groupings, meaning that records are assigned to
     * dom nodes in a linear manner. Unfortunately the position of
     * a dom node representing a record does no longer necessarily
     * correlate with the record's index in the underlying store when
     * grouping comes into play. Consider the following example:
     *
     * 1) Start with no grouping (all records belong to the same group A)
     *
     *    Order of records in the store
     *      1) recordA
     *      2) recordB
     *      3) recordC
     *
     *    Order of dom nodes in the rendered list (group A)
     *      1) dom node representing recordA
     *      2) dom node representing recordB
     *      3) dom node representing recordC
     *
     *  2) Move recordB to a new group B
     *
     *     Order of records in the store
     *       1) recordA
     *       2) recordB
     *       3) recordC
     *
     *     Order of dom nodes in the rendered list (group A rendered
     *     before group B !!!)
     *       1) dom node representing recordA -> group A
     *       2) dom node representing recordC -> group A
     *       3) dom node representing recordB -> group B
     *
     * Without this fix, recordB would be linked to the dom node which
     * actually represents recordC (!!!).
     *
     * Update:
     * The previous fix with overriding the viewRecordId, inspired by the ExtJS 4.2.0 updateIndex() method,
     * has caused a selection bug, where the wrong row was selected, after the grouping.
     * Consider the following example:
     *
     * 1) Start with no grouping (all rows belong to the same group A)
     *
     *    Order of records in the store
     *      a) rowA
     *      b) rowB
     *      c) rowC
     *      d) rowD
     *
     * 2) Group rows b) and c).
     * 3) After the grouping, selecting the rowD will select the grouped rowC and rowC will select the rowB.
     *
     * The reason for this bug is that is the method `getViewRange()` from AbstractView, was returning the records in
     * a order that was different (before grouping) from the order of records displayed by the grid (after grouping).
     * Consequently the call to `getRecord()` in AbstractView for the selected element, resulted to return the selected
     * record before the grouping (in AbstractView).
     *
     * The workaround solution was to concat the grouped records from the grid, and afterwards to clear the store data,
     * to get rid of the wrong record order, and afterwards to load the records back in the right order (after the grouping).
     * Afterwards, the `updateIndexes` method was called to map the viewRecordId with the correct store recordId
     * (is now a safe call, because the store data is sorted correctly).
     *
     * @see Ext.view.AbstractView.updateIndexes
     */
    viisonPickwareErpFixIndexes: function () {
        var groups = this.view.store.getGroups();
        var store = this.view.store;

        // Just to be safe, clear the selection model from the old records
        this.view.getSelectionModel().deselectAll();

        store.data.clear();
        store.suspendEvents();

        /**
         * Add the store data back in the correct order
         * NOTE: the methods `loadData()` and `loadRawData()` are not working for this case, because the stores
         * removed, updated and inserted property is not updated.
         * Why? From the code of `Ext.data.Store.add()` method, the store is calling the `insert()` method which creates
         * a new model (unlike the methods above) and inserts the new model in the store (probably the properties are set in that steps).
         * Also some additional events are called, which are probably queued, because of the `suspendEvents` method.
         * The index updating is also done in the `add()` method.
         */
        Ext.Array.each(groups, function (group) {
            Ext.Array.each(group.children, function (child) {
                store.data.add(child.internalId, child);
            });
        }, this);

        store.resumeEvents();
    },

});
