// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleBinLocationList.controller.List', {

    extend: 'Ext.app.Controller',

    listSelectorPrefix: '',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_article_bin_location_list/main',

    /**
     * @Override
     */
    init: function () {
        var controlDefinitions = {};
        controlDefinitions[this.listSelectorPrefix + ' viison_pickware_erp_article_bin_location_list-list'] = {
            editBinLocationMapping: this.onEditBinLocationMapping,
            selectAsDefaultBinLocationMapping: this.onSelectAsDefaultBinLocationMapping,
            deselectAsDefaultBinLocationMapping: this.onDeselectAsDefaultBinLocationMapping,
            groupclick: this.onGroupClick,
        };
        this.control(controlDefinitions);

        this.callParent(arguments);
    },

    /**
     * @param {Ext.view.Table} view
     * @param {HTMLElement} node
     * @param {String} groupName
     * @param {EventObject} e
     */
    onGroupClick: function (view, node, groupName, e) {
        var undoCollapse = true;
        switch (e.getTarget().getAttribute('data-action')) {
            case 'openWarehouse':
                this.openWarehouse(parseInt(groupName, 10));
                break;
            case 'editWarehouseStock':
                this.editWarehouseStock(view.panel, parseInt(groupName, 10), null);
                break;
            default:
                undoCollapse = false;
                break;
        }

        // Since we enabled the collapse action for the whole header row (collapsible=true), any click causes a
        // collapse. Since we do not want a button that has its own action to cause a collapse as well, we need to
        // collapse manually again to undo the unwanted collapse. Expand/collapse depending on the current state.
        if (undoCollapse) {
            var groupingFeature = view.panel.groupingFeature;
            if (groupingFeature.collapsedState[groupName]) {
                groupingFeature.expand(groupName);
            } else {
                groupingFeature.collapse(groupName);
            }
        }
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List} listPanel
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.model.BinLocationMapping} binLocationMapping
     */
    onEditBinLocationMapping: function (listPanel, binLocationMapping) {
        this.editWarehouseStock(
            listPanel,
            binLocationMapping.get('warehouseId'),
            binLocationMapping
        );
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List} listPanel
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.model.BinLocationMapping} binLocationMapping
     */
    onSelectAsDefaultBinLocationMapping: function (listPanel, binLocationMapping) {
        // Set 'defaultMapping' of the given bin location mapping and remove
        // it from all others in the same group (warehouse)
        var groupKey = binLocationMapping.store.getGroupString(binLocationMapping);
        var groupBinLocationMappings = binLocationMapping.store.getGroups(groupKey).children;
        Ext.Array.each(groupBinLocationMappings, function (mapping) {
            mapping.set('defaultMapping', mapping.get('id') === binLocationMapping.get('id'));
        });

        this.saveBinLocationMapping(
            listPanel,
            binLocationMapping,
            this.getViisonSnippet('notification/select_as_default_bin_location_mapping/success/title'),
            this.getViisonSnippet('notification/select_as_default_bin_location_mapping/success/message'),
            this.getViisonSnippet('notification/select_as_default_bin_location_mapping/failure/title'),
            this.getViisonSnippet('notification/select_as_default_bin_location_mapping/failure/message')
        );
    },

    /**
     * Remove the selection of binLocationMapping as the defaultMapping
     *
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List} listPanel
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.model.BinLocationMapping} binLocationMapping
     */
    onDeselectAsDefaultBinLocationMapping: function (listPanel, binLocationMapping) {
        // Unset 'defaultMapping' of the given bin location mapping
        binLocationMapping.set('defaultMapping', false);

        this.saveBinLocationMapping(
            listPanel,
            binLocationMapping,
            this.getViisonSnippet('notification/deselect_as_default_bin_location_mapping/success/title'),
            this.getViisonSnippet('notification/deselect_as_default_bin_location_mapping/success/message'),
            this.getViisonSnippet('notification/deselect_as_default_bin_location_mapping/failure/title'),
            this.getViisonSnippet('notification/deselect_as_default_bin_location_mapping/failure/message')
        );
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List} listPanel
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.model.BinLocationMapping} binLocationMapping
     * @param {String} successTitle
     * @param {String} successMessage
     * @param {String} failureTitle
     * @param {String} failureMessage
     */
    saveBinLocationMapping: function (
        listPanel,
        binLocationMapping,
        successTitle,
        successMessage,
        failureTitle,
        failureMessage
    ) {
        listPanel.getStore().sync({
            scope: this,
            success: function () {
                listPanel.getStore().group();
                Shopware.Notification.createGrowlMessage(
                    successTitle,
                    Ext.String.format(
                        successMessage,
                        binLocationMapping.get('binLocation').code,
                        binLocationMapping.get('articleNumber')
                    ),
                    'ViisonPickwareERP'
                );
            },
            failure: function () {
                binLocationMapping.reject();
                Shopware.Notification.createGrowlMessage(
                    failureTitle,
                    Ext.String.format(
                        failureMessage,
                        binLocationMapping.get('binLocation').code,
                        binLocationMapping.get('articleNumber')
                    ),
                    'ViisonPickwareERP'
                );
            },
        });
    },

    /**
     * @param {number} warehouseId
     */
    openWarehouse: function (warehouseId) {
        // Open a new warehouse management sub app for the warehouse
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.ViisonPickwareERPWarehouseManagement',
            params: {
                warehouseId: warehouseId,
            },
        });
    },

    /**
     * Opens a new stock editor window for the warehouse with the given warehouseId.
     *
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List} listPanel
     * @param {number} warehouseId
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.model.BinLocationMapping} selectedBinLocationMapping (optional)
     */
    editWarehouseStock: function (listPanel, warehouseId, selectedBinLocationMapping) {
        return this.getController('Shopware.apps.ViisonPickwareERPBinLocationEditor.controller.Main').createEditor(
            this.getArticleDetailId(listPanel),
            this.getDefaultPurchasePrice(listPanel),
            this.getWarehouse(listPanel, warehouseId),
            listPanel.store,
            this.createEditWarehouseStockCallback(listPanel),
            selectedBinLocationMapping
        );
    },

    /**
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List} listPanel
     * @param {number} warehouseId
     * @returns {object|null}
     */
    getWarehouse: function (listPanel, warehouseId) {
        var warehouse = null;
        Ext.each(listPanel.store.getGroups(), function (group) {
            if (group.name === warehouseId) {
                warehouse = group.children[0].get('warehouse');
            }
        });

        return warehouse;
    },

    /**
     * This method must be overwritten when using this component.
     *
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List} listPanel
     * @returns {number|null}
     */
    getArticleDetailId: function (listPanel) { // eslint-disable-line no-unused-vars
        return null;
    },

    /**
     * This method must be overwritten when using this component.
     *
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List} listPanel
     * @returns {number|null}
     */
    getDefaultPurchasePrice: function (listPanel) { // eslint-disable-line no-unused-vars
        return null;
    },

    /**
     * This method must be overwritten when using this component.
     *
     * @param {Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List} listPanel
     * @returns {function}
     */
    createEditWarehouseStockCallback: function (listPanel) { // eslint-disable-line no-unused-vars
        return Ext.emptyFn;
    },

});
