// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleBinLocationList.view.List', {

    extend: 'Ext.grid.Panel',
    alias: 'widget.viison_pickware_erp_article_bin_location_list-list',
    cls: 'viison_pickware_erp_article_bin_location_list-list viison-common--grid',

    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_article_bin_location_list/main',

    split: true,
    border: true,

    // Remark: Do not set autoscroll (default: false). This disables scrollbars (like overflow:hidden),
    // but since the grid scales automatically without any restriction in height, we do not need scrollbars
    // anyway. This fixes unwanted scrollbars while zoomed in.
    viewConfig: {
        enableTextSelection: true,
        markDirty: false,
    },
    bodyStyle: {
        padding: '0 0 15px 0',
        background: '#FFFFFF',
    },
    collapseGroupsUponRefresh: false,
    hasMultiRowHeader: false,
    STOCK_COLUMN_WIDTH: 120,
    ACTION_COLUMN_WIDTH: 145,

    /**
     * @Override
     */
    initComponent: function () {
        this.columns = this.createColumns();
        this.groupingFeature = this.createGroupingFeature();
        this.features = [
            this.groupingFeature,
        ];
        this.store = this.createStore();

        this.callParent(arguments);
    },

    /**
     * @returns {Shopware.apps.ViisonPickwareERPArticleBinLocationList.store.BinLocationMappings}
     */
    createStore: function () {
        // Create the grouped store
        return Ext.create('Shopware.apps.ViisonPickwareERPArticleBinLocationList.store.BinLocationMappings', {
            listeners: {
                scope: this,
                refresh: function (store) {
                    if (this.collapseGroupsUponRefresh) {
                        // Mark all but the first group as collapsed
                        this.groupingFeature.collapsedState = {};
                        for (var i = 1; i < store.getGroups().length; i += 1) {
                            this.groupingFeature.collapsedState[store.getGroups()[i].name] = true;
                        }
                        this.collapseGroupsUponRefresh = false;
                    }
                },
            },
        });
    },

    /**
     * @return Ext.grid.column.Column[]
     */
    createColumns: function () {
        return [
            {
                dataIndex: 'isMappingToNullBinLocation',
                header: this.getViisonSnippet('list/column/bin_location'),
                sortable: false,
                flex: 1,
                renderer: function (value, metaData, record) {
                    metaData.style = metaData.style || '';
                    metaData.style += 'padding-left: 25px;';

                    var nullBinLocationInfo = this.getViisonSnippet('list/column/bin_location/null_bin_location');
                    var renderValue = value ? '<i>' + nullBinLocationInfo + '</i>' : record.get('binLocation').code;

                    if (record.get('defaultMapping')) {
                        var tooltip = this.getViisonSnippet('list/column/bin_location/default_bin_location_mapping');
                        renderValue += '<img alt="" src="data:image/gif;base64,R0lGODlhAQABAID/AMDAwAAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==" class="sprite-star" style="margin-bottom: -4px; margin-left: 5px; padding-left: 16px;" data-qtip="' + tooltip + '">';
                    }

                    return renderValue;
                },
            }, {
                header: this.getViisonSnippet('list/column/minimum_stock'),
                sortable: false,
                width: this.STOCK_COLUMN_WIDTH,
                align: 'right',
            }, {
                header: this.getViisonSnippet('list/column/target_stock'),
                sortable: false,
                width: this.STOCK_COLUMN_WIDTH,
                align: 'right',
            }, {
                dataIndex: 'stock',
                header: this.getViisonSnippet('list/column/stock'),
                sortable: false,
                width: this.STOCK_COLUMN_WIDTH,
                align: 'right',
            }, {
                xtype: 'actioncolumn',
                width: this.ACTION_COLUMN_WIDTH,
                items: this.createActions(),
            },
        ];
    },

    /**
     * @return Object[]
     */
    createActions: function () {
        return [
            {
                iconCls: 'sprite-pencil',
                action: 'editBinLocationMapping',
                tooltip: this.getViisonSnippet('list/action/edit_bin_location_mapping'),
                scope: this,
                handler: function (grid, rowIndex) {
                    this.fireEvent('editBinLocationMapping', this, grid.getStore().getAt(rowIndex));
                },
            }, {
                iconCls: 'sprite-star--plus',
                action: 'selectAsDefaultBinLocationMapping',
                tooltip: this.getViisonSnippet('list/action/select_as_default_bin_location_mapping'),
                scope: this,
                getClass: function (value, meta, record) {
                    return (record.get('defaultMapping') || record.get('isMappingToNullBinLocation')) ? 'x-hide-display' : '';
                },
                handler: function (grid, rowIndex) {
                    this.fireEvent('selectAsDefaultBinLocationMapping', this, grid.getStore().getAt(rowIndex));
                },
            }, {
                iconCls: 'sprite-star--minus',
                action: 'deselectAsDefaultBinLocationMapping',
                tooltip: this.getViisonSnippet('list/action/deselect_as_default_bin_location_mapping'),
                scope: this,
                getClass: function (value, meta, record) {
                    return (record.get('defaultMapping') && !record.get('isMappingToNullBinLocation')) ? '' : 'x-hide-display';
                },
                handler: function (grid, rowIndex) {
                    this.fireEvent('deselectAsDefaultBinLocationMapping', this, grid.getStore().getAt(rowIndex));
                },
            }, {
                iconCls: 'sprite-sticky-notes-pin',
                action: 'openBinLocation',
                tooltip: this.getViisonSnippet('list/action/open_bin_location'),
                scope: this,
                getClass: function (value, meta, record) {
                    return (record.get('isMappingToNullBinLocation')) ? 'x-hide-display' : '';
                },
                handler: function (grid, rowIndex) {
                    var binLocationMapping = grid.getStore().getAt(rowIndex);
                    Shopware.app.Application.addSubApplication({
                        name: 'Shopware.apps.ViisonPickwareERPWarehouseManagement',
                        params: {
                            warehouseId: binLocationMapping.get('warehouseId'),
                            binLocationCode: binLocationMapping.get('binLocation').code,
                        },
                    });
                },
            },
        ];
    },

    /**
     * Creates the grid's grouping feature with custom header.
     *
     * @returns {ViisonPickwareERP.Ext.grid.feature.Grouping}
     */
    createGroupingFeature: function () {
        // Add a grouping feature with custom header
        var groupingFeature = Ext.create('ViisonPickwareERP.Ext.grid.feature.Grouping', {
            hideGroupedHeader: false,
            enableGroupingMenu: false,
            // The collapsible property allows the whole header row to collapse the warehouse, including buttons
            // that have a separate functions. The controller will handle these unwanted collapses.
            collapsible: true,
            groupHeaderTpl: this.createGroupHeaderTemplate(),
        });

        return groupingFeature;
    },

    /**
     * @return Ext.XTemplate|Array
     */
    createGroupHeaderTemplate: function () {
        /* eslint-disable indent */
        return new Ext.XTemplate(
            '<div class="is--warehouse-group-header {children:this.getDisplayMode}">',
                '<div class="is--warehouse-name-panel">',
                    '<span class="is--group-title">',
                        '{children:this.getWarehouseDisplayName}',
                    '</span>',
                    '<span class="sprite-sticky-notes-pin is--open-warehouse-button" data-action="openWarehouse" ',
                         'data-qtip="' + this.getViisonSnippet('list/group_header/tooltip/open_warehouse') + '"></span>',
                    '<span class="is--single-row-stock-available-info">',
                        '<span class="is--stock-available-label">',
                            this.getViisonSnippet('list/group_header/stock_available'),
                        '</span>',
                        '<span class="{children:this.getStockIcon} is--stock-available-icon"></span>',
                    '</span>',
                '</div>',
                '<div>',
                    '<div class="is--stock-levels-panel" style="width: ' + ((this.STOCK_COLUMN_WIDTH * 3) + this.ACTION_COLUMN_WIDTH) + 'px;">',
                        '<span class="is--stock-column" style="width: ' + this.STOCK_COLUMN_WIDTH + 'px;">',
                            '{children:this.getWarehouseMinimumStock}',
                        '</span>',
                        '<span class="is--stock-column" style="width: ' + this.STOCK_COLUMN_WIDTH + 'px;">',
                            '{children:this.getWarehouseTargetStock}',
                        '</span>',
                        '<span class="is--stock-column" style="width: ' + this.STOCK_COLUMN_WIDTH + 'px;">',
                            '{children:this.getWarehousePhysicalStock}',
                        '</span>',
                        '<span class="is--stock-edit-button" style="width:' + this.ACTION_COLUMN_WIDTH + 'px;">',
                            '<div class="x-btn primary small x-toolbar-item x-btn-default-small x-noicon x-btn-noicon x-btn-default-small-noicon" style="padding: 2px 5px;">',
                                '<em>',
                                    '<button class="x-btn-center" hidefocus="true" role="button" data-action="editWarehouseStock">',
                                        '<span class="x-btn-inner" data-action="editWarehouseStock">',
                                            this.getViisonSnippet('list/group_header/button/edit_stock'),
                                        '</span>',
                                    '</button>',
                                '</em>',
                            '</div>',
                        '</span>',
                    '</div>',
                    '<span class="is--multi-row-stock-available-info">',
                        '<span class="is--stock-available-label">',
                            this.getViisonSnippet('list/group_header/stock_available'),
                        '</span>',
                        '<span class="{children:this.getStockIcon} is--stock-available-icon"></span>',
                    '</span>',
                '</div>',
            '</div>',
            {
                getDisplayMode: function () {
                    return this.hasMultiRowHeader ? 'is--multi-row' : '';
                }.bind(this),
                getWarehouseDisplayName: function (children) {
                    return (children && children.length > 0) ? Ext.String.format('[0] ([1])', children[0].get('warehouse').name, children[0].get('warehouse').code) : '';
                },
                getStockIcon: function (children) {
                    return (children && children.length > 0 && children[0].get('warehouse').stockAvailableForSale) ? 'sprite-tick' : 'sprite-cross';
                },
                getWarehouseMinimumStock: function (children) {
                    return (children && children.length > 0) ? children[0].get('articleDetailConfigurations').minimumStock : '-';
                },
                getWarehouseTargetStock: function (children) {
                    return (children && children.length > 0) ? children[0].get('articleDetailConfigurations').targetStock : '-';
                },
                getWarehousePhysicalStock: function (children) {
                    if (!children || children.length === 0) {
                        return 0;
                    }

                    return children.reduce(function (sum, record) {
                        return sum + record.get('stock');
                    }, 0);
                },
            }
        );
        /* eslint-enable indent */
    },

});
