// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleListBarcodeLabelPrinting.controller.Grid', {

    extend: 'Ext.app.Controller',
    refs: [
        { ref: 'mainGrid', selector: 'multi-edit-main-grid' },
    ],
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_article_list_barcode_label_printing/main',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'multi-edit-main-grid': {
                viisonPickwareERPAddArticleLabel: this.onAddArticleLabelsButtonClicked,
            },
        });

        this.callParent(arguments);
    },

    /**
     * @param Shopware.apps.Article.model.Article[] records
     * @param Ext.grid.Panel grid
     * @param Ext.button.Button button
     */
    onAddArticleLabelsButtonClicked: function (records, grid, button) {
        if (records.length === 0) {
            // If no articles where selected, use all that the filter applies to after asking the user for confirmation
            Ext.Msg.confirm(this.getViisonSnippet('confirmation_message/add_all_articles/title'), this.getViisonSnippet('confirmation_message/add_all_articles/text'), function (buttonId) {
                if (buttonId !== 'yes') {
                    return;
                }

                grid.getView().loadMask.show();
                button.setDisabled(true);

                this.sendAddBarcodeAjaxRequest(
                    ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBarcodeLabelPrinting/addAllFilteredArticleBarcodeLabels'),
                    {
                        productFilter: grid.getStore().getProxy().extraParams.ast,
                    }
                );
            }, this);
        } else {
            // The user selected articles, use those
            this.addArticleLabels(records);
        }
    },

    /**
     * @param Shopware.apps.Article.model.Article[] records
     */
    addArticleLabels: function (records) {
        var articleNumbers = [];
        // Collect the article numbers of all selected records
        Ext.each(records, function (record) {
            articleNumbers.push(record.get('Detail_number'));
        });

        this.sendAddBarcodeAjaxRequest(
            ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPBarcodeLabelPrinting/enqueueBarcodeLabel'),
            {
                type: 'article',
                identifiers: articleNumbers,
            }
        );
    },

    /**
     * @param String url
     * @param Object data
     */
    sendAddBarcodeAjaxRequest: function (url, data) {
        // Increase AJAX timeout temporarily to 5 minutes (this process may take a while when dealing with a lot of articles)
        Ext.Ajax.timeout = 300000;
        Ext.Ajax.request({
            method: 'POST',
            url: url,
            jsonData: data,
            scope: this,
            callback: function (options, success, response) {
                if (success) {
                    if (response && response.responseText) {
                        try {
                            var responseData = JSON.parse(response.responseText);
                            success = responseData.success;
                        } catch (e) {
                            success = false;
                        }
                    } else {
                        success = false;
                    }
                }

                // Display a notification
                var title = (success) ? this.getViisonSnippet('notification/add_article_barcode_labels/success/title') : this.getViisonSnippet('notification/add_article_barcode_labels/error/title');
                var message = (success) ? this.getViisonSnippet('notification/add_article_barcode_labels/success/text') : this.getViisonSnippet('notification/add_article_barcode_labels/error/text');
                Shopware.Msg.createGrowlMessage(title, message, 'ViisonPickwareERPArticleListBarcodeLabelPrinting');

                // Hide the load mask by reloading the store (directly calling this.getMainGrid().getView().loadMask.hide() makes part of the grid unusable for some reason)
                this.getMainGrid().getStore().load();
                this.getMainGrid().down('[action=viisonPickwareERPAddArticleLabel]').setDisabled(false);

                // Reset the timeout
                Ext.Ajax.timeout = 30000;
            },
        });
    },

});
