// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleOrders.controller.Main', {

    extend: 'Ext.app.Controller',
    refs: [
        { ref: 'articleOrdersList', selector: 'viison_pickware_erp_article_orders-detail-article_orders_list' },
        { ref: 'fromField', selector: 'viison_pickware_erp_article_orders-detail-article_orders_list toolbar datefield[name=from]' },
        { ref: 'toField', selector: 'viison_pickware_erp_article_orders-detail-article_orders_list toolbar datefield[name=to]' },
    ],

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_article_orders-detail-article_orders_list': {
                showCustomer: this.onShowCustomer,
                showOrder: this.onShowOrder,
                afterrender: this.onOrdersListAfterRender,
            },
            'viison_pickware_erp_article_orders-detail-article_orders_list viison_common_variant_combo_box-combo_box': {
                variantChanged: this.onVariantChanged,
            },
            'viison_pickware_erp_article_orders-detail-article_orders_list toolbar datefield': {
                change: this.onFilterChange,
            },
            'viison_pickware_erp_article_orders-detail-article_orders_list toolbar button[action=export]': {
                click: this.onExportButtonClicked,
            },
        });

        // Create the orders store
        this.ordersStore = Ext.create('Shopware.apps.ViisonPickwareERPArticleOrders.store.ArticleOrder', {});

        this.callParent(arguments);
    },

    /**
     * Creates and returns a new article orders grid panel.
     *
     * @return Shopware.apps.ViisonPickwareERPArticleOrders.view.detail.ArticleOrdersList
     */
    createArticleOrdersTab: function () {
        var tab = Ext.create('Shopware.apps.ViisonPickwareERPArticleOrders.view.detail.ArticleOrdersList', {
            store: this.ordersStore,
        });

        this.subApplication.on('ProductModule:storesChanged', this.articleChanged, tab);

        return tab;
    },

    /**
     * Ensures that the data isn't loaded before the grid is actually rendered.
     * This ensures that the empty text message will always be displayed if necessary,
     * what wouldn't be the case if the store is loaded before the grid is rendered.
     */
    onOrdersListAfterRender: function () {
        setTimeout(function () {
            // Pass the article to the variant selection to load its data
            var variantSelection = this.getArticleOrdersList().down('viison_common_variant_combo_box-combo_box');
            variantSelection.setArticle(this.article);
        }.bind(this), 50);
    },

    /**
     * If given an article detail, the respective orders are loaded, otherwise the
     * store and its filters are cleared.
     *
     * @param Shopware.apps.Article.model.Detail|null|false articleDetail
     */
    onVariantChanged: function (articleDetail) {
        this.selectedVariant = articleDetail;
        this.loadOrders();
    },

    /**
     * Reloads the orders store, if the filter form is valid.
     */
    onFilterChange: function () {
        this.loadOrders();
    },

    /**
     * Opens a new browser window/tab showing the exported orders matching the currently
     * applied filter.
     */
    onExportButtonClicked: function () {
        // Create the URL using the current store filters
        var filter = Ext.JSON.encode(this.ordersStore.filters.items.map(function (item) {
            return {
                property: item.property,
                value: item.value,
                expression: (item.expression) ? item.expression : null,
                operator: (item.operator) ? item.operator : null,
            };
        }));
        var url = ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPArticleOrders/exportArticleOrdersList', {
            filter: filter,
        });

        window.open(url, '_blank');
    },

    /**
     * Opens a new Customer subapp showing the customer associated with the given record.
     *
     * @param Shopware.apps.ViisonPickwareERPArticleOrders.model.ArticleOrder record
     */
    onShowCustomer: function (record) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Customer',
            action: 'detail',
            params: {
                customerId: record.get('userId'),
            },
        });
    },

    /**
     * Opens a new Order subapp showing the order associated with the given record.
     *
     * @param Shopware.apps.ViisonPickwareERPArticleOrders.model.ArticleOrder record
     */
    onShowOrder: function (record) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Order',
            action: 'detail',
            params: {
                orderId: record.get('orderId'),
            },
        });
    },

    /**
     * Saves the article and adds its ID as a new filter to the main orders store.
     *
     * @param Shopware.apps.Article.model.Article article
     */
    setArticle: function (article) {
        this.article = article;

        // Add the article ID as a filter to the store
        this.ordersStore.filters.add('articleId', Ext.create('Ext.util.Filter', {
            property: 'orderDetail.articleId',
            value: this.article.get('id'),
        }));
    },

    /**
     * Validates the filters and, if valid, reloads the orders.
     */
    loadOrders: function () {
        // Update filters
        if (this.selectedVariant) {
            this.ordersStore.filters.add('articleNumber', Ext.create('Ext.util.Filter', {
                property: 'orderDetail.articleNumber',
                value: this.selectedVariant.get('number'),
            }));
        } else {
            this.ordersStore.filters.removeAtKey('articleNumber');
        }
        if (this.getFromField().getValue()) {
            this.ordersStore.filters.add('from', Ext.create('Ext.util.Filter', {
                property: 'order_.orderTime',
                value: this.getFromField().getValue(),
                expression: '>=',
            }));
        } else {
            this.ordersStore.filters.removeAtKey('from');
        }
        if (this.getToField().getValue()) {
            // Add one day to the date to include the chosen date
            var toDate = new Date(this.getToField().getValue().getTime() + (24 * 60 * 60 * 1000));
            this.ordersStore.filters.add('to', Ext.create('Ext.util.Filter', {
                property: 'order_.orderTime',
                value: toDate,
                expression: '<',
            }));
        } else {
            this.ordersStore.filters.removeAtKey('to');
        }

        this.ordersStore.loadPage(1);
    },

    /**
     * This method gets called when the user switches the article of the detail window by clicking on an article in
     * the article list while the split view mode is active.
     *
     * @param Shopware.apps.Article.model.Article article
     */
    articleChanged: function (article) {
        this.down('viison_common_variant_combo_box-combo_box').setArticle(article);
    },

});
