// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleProfitMargin.view.detail.Prices', {

    override: 'Shopware.apps.Article.view.detail.Prices',

    /**
     * @Override
     * Add listener for purchase price change event
     */
    initComponent: function () {
        this.currentPurchasePrice = 0.0;

        // Listen to updates of 'viisonPurchasePrice' and the main article's 'tax'
        this.subApp.articleWindow.on('viisonPurchasePriceChanged', this.refreshPurchasePriceAndView, this);
        this.subApp.articleWindow.on('articleTaxChanged', this.onArticleTaxChanged, this);

        // Remark (case: edit variant): if we do not unregister this event, a reference to this price-grid will be kept, even if it is closed.
        // An new event (fired in a second edit variant) would cause a NullPointerException, because there is no active tab in this closed price-grid.
        // Therefore remove this listener with its reference when the price-grid is closed.
        this.on('beforedestroy', function () {
            this.subApp.articleWindow.un('viisonPurchasePriceChanged', this.refreshPurchasePriceAndView, this);
            this.subApp.articleWindow.un('articleTaxChanged', this.onArticleTaxChanged, this);
        }, this);

        this.callParent(arguments);
    },

    /**
     * @Override
     */
    onStoresLoaded: function (article, stores) {
        // Save tax store
        this.taxStore = stores.taxes;

        // Save the article in 'mainArticle', because 'this.article' can either be an article
        // or an article detail, depending on where this component is used. That said, in case
        // this componen is used in the variant editing, 'this.article' is an article detail and
        // 'mainArticle' will be overridden with the correct article, before it is used.
        this.mainArticle = article;

        this.callParent(arguments);
    },

    /**
     * Saves the changed purchase price and re-calculates the profit margins.
     *
     * @param field
     * @param newValue
     */
    refreshPurchasePriceAndView: function (field, newValue) {
        this.currentPurchasePrice = newValue;
        this.calculateMargins();
    },

    /**
     * Saves the changed tax rate and re-calculates the profit margins.
     *
     * @param Ext.form.field.ComboBox field
     * @param Shopware.apps.Base.model.Tax newValue
     */
    onArticleTaxChanged: function (field, newValue) {
        this.subApp.selectedTaxRate = (newValue) ? newValue.get('tax') : 0.0;
        this.calculateMargins();
    },

    /**
     * @Override
     * Add listener to price-column and profit-margin-column
     *
     * @returns tabPanel
     */
    createElements: function () {
        var tabPanel = this.callParent(arguments);

        // Fire event, which allows other views/controllers to react on the preparation
        // of the price store
        this.subApp.articleWindow.fireEvent('didCreateElements', this);

        // Determine default tax rate, if no tax rate has been selected yet
        if (typeof this.subApp.selectedTaxRate === 'undefined') {
            var taxIndex = this.taxStore.findExact('id', this.mainArticle.get('taxId'));
            this.subApp.selectedTaxRate = (taxIndex > -1) ? this.taxStore.getAt(taxIndex).get('tax') : 0.0;
        }

        Ext.Array.each(this.priceGrids, function (tab) {
            Ext.Array.each(tab.columns, function (column) {
                if (column.dataIndex === 'price') {
                    var priceListeners = column.editor.listeners || {};
                    priceListeners.change = function (field, newValue) {
                        // Save new price an calculate the margin
                        var record = column.up('grid').getSelectionModel().getSelection()[0];
                        record.set('price', newValue);
                        this.updateProfitMargin(record);

                        // Commit changes to remove red 'dirty' flags and refresh the grid
                        record.store.commitChanges();
                        tab.getView().refresh();
                    }.bind(this);
                    column.editor.listeners = priceListeners;
                } else if (column.dataIndex === 'viisonProfitMargin') {
                    var profitMarginListeners = column.editor.listeners || {};
                    profitMarginListeners.change = function (field, newValue) {
                        // Save new margin an calculate the price
                        var record = column.up('grid').getSelectionModel().getSelection()[0];
                        record.set('viisonProfitMargin', newValue);
                        this.updatePrice(record);

                        // Commit changes to remove red 'dirty' flags and refresh the grid
                        record.store.commitChanges();
                        tab.getView().refresh();
                    }.bind(this);
                    column.editor.listeners = profitMarginListeners;
                }
            }, this);
        }, this);

        // Add a listener on the tab panel to be able to calculate the profit margins for the
        // prices contained in the store, once the tab (and the prices in the store) changes
        tabPanel.on('beforetabchange', function () {
            this.calculateMargins();
        }, this);

        return tabPanel;
    },

    /**
     * @Override
     * Add profit-margin-column
     *
     * @returns columns
     */
    getColumns: function () {
        var columns = this.callParent(arguments);

        // Find the index of the price column and check for a profit margin column to make sure
        // to not add it twice
        var priceIndex = -1;
        var profitMarginColumnExists = false;
        Ext.Array.each(columns, function (column, index) {
            if (column.dataIndex === 'price') {
                priceIndex = index;
            } else if (column.dataIndex === 'viisonProfitMargin') {
                profitMarginColumnExists = true;

                return false;
            }

            return undefined;
        });
        if (profitMarginColumnExists) {
            return columns;
        }

        // Create and add the profit margin column right after the price column
        columns.splice((priceIndex + 1), 0, {
            header: ViisonCommonApp.getSnippet(
                'column/profitmargin/header',
                'backend/viison_pickware_erp_article_profit_margin/main'
            ),
            tooltip: ViisonCommonApp.getSnippet(
                'column/profitmargin/tooltip',
                'backend/viison_pickware_erp_article_profit_margin/main'
            ),
            dataIndex: 'viisonProfitMargin',
            renderer: function (value, metaData, record) {
                var margin = record.get('viisonProfitMargin');
                var formattedMargin = Ext.util.Format.number(margin, '0.0') + ' %';
                if (margin <= 0) {
                    return "<span style='color:red' >" + formattedMargin + '</span>';
                }

                return formattedMargin;
            },
            editor: {
                xtype: 'numberfield',
                maxValue: 99.9,
            },
        });

        return columns;
    },

    /**
     * Calculates the profit margin for each price currently contained in the priceStore using
     * 'updateProfitMargin()'.
     */
    calculateMargins: function () {
        // Calculate the profit margins of all prices
        this.priceStore.each(function (price) {
            this.updateProfitMargin(price);
        }, this);

        // Commit changes to remove red 'dirty' flags
        this.priceStore.commitChanges();
    },

    /**
     * Calculates the profit margin for the given price using the selling price as well as the current
     * purchase price. Since the profit margin can only be calculated if both prices are of the same type,
     * either 'gross' or 'net', the selling price is converted based on the config of the customer group
     * and the purchase price mode. That is, if the selling price must be entered as 'net' and the purchase
     * price as 'gross', the selling price is converted to 'gross'. If the selling price must be entered as
     * 'gross' and the purchase price as 'net', the selling price is converted to 'net'.
     *
     * @param Shopware.apps.Article.model.Price price
     */
    updateProfitMargin: function (price) {
        // Determine the selling price in the same mode as the purchase price (gross or net)
        var sellingPrice = price.get('price');
        if (this.expectsNetPriceAsInput(price) && !ViisonPickwarePurchasePriceHelper.purchasePriceIsNet()) {
            // Convert the price from net to gross
            sellingPrice *= 1.0 + (this.subApp.selectedTaxRate / 100.0);
        } else if (!this.expectsNetPriceAsInput(price) && ViisonPickwarePurchasePriceHelper.purchasePriceIsNet()) {
            // Convert the price from gross to net
            sellingPrice /= 1.0 + (this.subApp.selectedTaxRate / 100.0);
        }

        // Prevent division by 0, e.g. when creating a new article
        var margin = (sellingPrice === 0.0) ? 0.0 : (100 * ((sellingPrice - this.currentPurchasePrice) / sellingPrice));

        price.set('viisonProfitMargin', margin);
    },

    /**
     * Calculates the selling price for the given price using the profit margin as well as the current
     * purchase price. Since the profit margin was calculate using either two 'gross' or two 'net' values,
     * the calculated selling price might be converted after the main calculatin, based on the config of the
     * customer group and the purchase price mode. That is, if the selling price must be entered as 'net'
     * and the purchase price as 'gross', the selling price is converted to 'net'. If the selling price must
     * be entered as 'gross' and the purchase price as 'net', the selling price is converted to 'gross'.
     *
     * @param Shopware.apps.Article.model.Price price
     */
    updatePrice: function (price) {
        // Calculate the selling price in the same mode as the purchase price (gross or net)
        var sellingPrice = this.currentPurchasePrice / (1 - (price.get('viisonProfitMargin') / 100));

        // Convert the selling price in the mode configured in the price's customer group (gross or net)
        if (this.expectsNetPriceAsInput(price) && !ViisonPickwarePurchasePriceHelper.purchasePriceIsNet()) {
            // Convert the price from gross to net
            sellingPrice /= 1.0 + (this.subApp.selectedTaxRate / 100.0);
        } else if (!this.expectsNetPriceAsInput(price) && ViisonPickwarePurchasePriceHelper.purchasePriceIsNet()) {
            // Convert the price from net to gross
            sellingPrice *= 1.0 + (this.subApp.selectedTaxRate / 100.0);
        }

        price.set('price', sellingPrice);
    },

    /**
     * Checks whether the customer group of the given price requires input of net prices.
     *
     * @param Shopware.apps.Article.model.Price price
     * @return boolean
     */
    expectsNetPriceAsInput: function (price) {
        var customerGroupIndex = this.customerGroupStore.findExact('key', price.get('customerGroupKey'));
        var customerGroup = (customerGroupIndex > -1) ? this.customerGroupStore.getAt(customerGroupIndex) : null;

        return customerGroup && !customerGroup.get('taxInput');
    },

});
