// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleStock.controller.Main', {

    extend: 'Ext.app.Controller',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_article_stock/main',

    /**
     * @Override
     */
    init: function () {
        this.control({
            'viison_pickware_erp_article_stock-detail-stock_list': {
                typeChanged: this.onTypeChanged,
                saveStockEntry: this.onSaveStockEntry,
                showOrder: this.onShowOrder,
                showSupplierOrder: this.onShowSupplierOrder,
                afterrender: this.onAfterRender,
            },
            'viison_pickware_erp_article_stock-detail-stock_list viison_common_variant_combo_box-combo_box': {
                variantChanged: this.onVariantChanged,
            },
            'viison_pickware_erp_article_stock-detail-stock_list viison_pickware_erp_warehouse_management-warehouse_combo_box': {
                warehouseChanged: this.onWarehouseChanged,
            },
        });

        // Create the main stock store as well as a store for available stock types
        this.stockStore = Ext.create('Shopware.apps.ViisonPickwareERPArticleStock.store.Stock', {});
        this.stockTypeStore = Ext.create('Shopware.apps.ViisonPickwareERPArticleStock.store.StockType', {});

        this.callParent(arguments);
    },

    /**
     * Creates and returns a new stock grid panel.
     *
     * @return Shopware.apps.ViisonPickwareERPArticleStock.view.detail.StockList
     */
    createArticleStockListTab: function () {
        this.viisonPickwareERPStockListTab = Ext.create('Shopware.apps.ViisonPickwareERPArticleStock.view.detail.StockList', {
            store: this.stockStore,
            stockTypeStore: this.stockTypeStore,
        });

        this.subApplication.on('ProductModule:storesChanged', this.articleChanged, this.viisonPickwareERPStockListTab);

        return this.viisonPickwareERPStockListTab;
    },

    /**
     * Ensures that the data isn't loaded before the grid is actually rendered. This ensures that the empty text message
     * will always be displayed if necessary, what wouldn't be the case if the store is loaded before the grid is
     * rendered.
     */
    onAfterRender: function () {
        setTimeout(function () {
            // Pass the article to the variant selection to load its data
            var variantSelection = this.viisonPickwareERPStockListTab.down('viison_common_variant_combo_box-combo_box');
            variantSelection.setArticle(this.article);
        }.bind(this), 50);
    },

    setArticle: function (article) {
        this.article = article;
    },

    /**
     * Saves the given type and reloads the stock store.
     *
     * @param String type
     */
    onTypeChanged: function (type) {
        this.selectedType = type;
        this.loadStocks();
    },

    /**
     * Saves the given variant and reloads the stock store.
     *
     * @param Shopware.apps.Article.model.Detail|null|false articleDetail
     */
    onVariantChanged: function (articleDetail) {
        this.selectedVariant = articleDetail;
        this.loadStocks();

        // Update the stock list tab
        if (this.selectedVariant && this.selectedVariant.get('pickwareStockManagementDisabled')) {
            this.viisonPickwareERPStockListTab.gridPanel.disable();
            this.viisonPickwareERPStockListTab.notStockManagedInfo.show();
        } else {
            this.viisonPickwareERPStockListTab.gridPanel.enable();
            this.viisonPickwareERPStockListTab.notStockManagedInfo.hide();
        }
    },

    /**
     * Saves the given warehouse and reloads the stock store.
     *
     * @param Shopware.apps.ViisonPickwareERPWarehouseManagement.model.Warehouse|null|false warehouse
     */
    onWarehouseChanged: function (warehouse) {
        this.selectedWarehouse = warehouse;
        this.loadStocks();
    },

    /**
     * Saves the passed stock record in the database. If the operation was successful and the request's
     * 'updateLotAssignedEntries' parameter is true, the store is reloaded to fetch the updated lot assigned
     * entries data.
     *
     * @param Shopware.apps.ViisonPickwareERPArticleStock.model.Stock record
     * @param Boolean skipAlert
     */
    onSaveStockEntry: function (record, skipAlert) {
        if (skipAlert !== true) {
            // Ask the user whether lot assigned entries of the edited record should be updated too
            Ext.Msg.confirm(
                this.getViisonSnippet('alert/save_stock_entry/title'),
                this.getViisonSnippet('alert/save_stock_entry/message'),
                function (buttonId) {
                    record.getProxy().extraParams.updateLotAssignedEntries = buttonId === 'yes';
                    this.onSaveStockEntry(record, true);
                },
                this
            );

            return;
        }

        // Save the changes
        record.save({
            callback: function (data, operation) {
                if (operation.success) {
                    if (record.getProxy().extraParams.updateLotAssignedEntries) {
                        // Reload the store to fetch the updated lot assigned entries data
                        record.store.load();
                    }
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/save_stock_entry/success/title'),
                        this.getViisonSnippet('notification/save_stock_entry/success/message'),
                        'Pickware'
                    );
                } else {
                    Shopware.Notification.createGrowlMessage(
                        this.getViisonSnippet('notification/save_stock_entry/failure/title'),
                        this.getViisonSnippet('notification/save_stock_entry/failure/message'),
                        'Pickware'
                    );
                }
            }.bind(this),
        });
    },

    /**
     * Opens a new order subapp showing the order or return shipment associated with the given record.
     *
     * @param {Shopware.apps.ViisonPickwareERPArticleStock.model.Stock} stockEntry
     */
    onShowOrder: function (stockEntry) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.Order',
            params: {
                orderId: stockEntry.get('orderId'),
                returnShipmentId: stockEntry.get('returnShipmentId'),
            },
        });
    },

    /**
     * Opens a new supplier orders subapp showing the order associated with the given record.
     *
     * @param {Shopware.apps.ViisonPickwareERPArticleStock.model.Stock} stockEntry
     */
    onShowSupplierOrder: function (stockEntry) {
        Shopware.app.Application.addSubApplication({
            name: 'Shopware.apps.ViisonPickwareERPSupplierOrders',
            params: {
                orderId: stockEntry.get('supplierOrderId'),
            },
        });
    },

    /**
     * Updates the filters on the stock store and reloads it.
     */
    loadStocks: function () {
        // There might be no variant or warehouse selected during the loading process - only load
        // the stocks after a variant and a warehouse has been selected
        if (!this.selectedVariant || !this.selectedWarehouse) {
            return;
        }

        // Clear the store, if the selected variant is not stock managed
        if (this.selectedVariant.get('pickwareStockManagementDisabled')) {
            this.stockStore.clearFilter(true);
            this.stockStore.loadData([]);

            return;
        }

        // Update filters
        if (this.selectedType) {
            this.stockStore.filters.add('type', Ext.create('Ext.util.Filter', {
                property: 'stockLedgerEntry.type',
                value: this.selectedType,
            }));
        } else {
            this.stockStore.filters.removeAtKey('type');
        }

        this.stockStore.filters.add('articleDetailId', Ext.create('Ext.util.Filter', {
            property: 'stockLedgerEntry.articleDetailId',
            value: this.selectedVariant.get('id'),
        }));
        this.stockStore.filters.add('warehouseId', Ext.create('Ext.util.Filter', {
            property: 'stockLedgerEntry.warehouseId',
            value: this.selectedWarehouse.get('id'),
        }));

        this.stockStore.loadPage(1);
    },

    /**
     * This method gets called when the user switches the article of the detail window by clicking on an article in
     * the article list while the split view mode is active.
     *
     * @param Shopware.apps.Article.model.Article article
     */
    articleChanged: function (article) {
        this.article = article;
        this.down('viison_common_variant_combo_box-combo_box').setArticle(article);
    },

});
