// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleStock.view.detail.StockList', {

    extend: 'Ext.panel.Panel',
    alias: 'widget.viison_pickware_erp_article_stock-detail-stock_list',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_article_stock/main',

    layout: 'fit',
    border: false,
    style: {
        borderBottom: '1px solid #a4b5c0',
    },
    bodyStyle: {
        border: 'none',
    },

    /**
     * @Override
     */
    initComponent: function () {
        // Create a hidden info box for articles, whose stock is not managed
        this.notStockManagedInfo = Ext.create('Ext.container.Container', {
            xtype: 'container',
            cls: 'viison-common--info-panel',
            html: this.getViisonSnippet('toolbar/pickware_stock_management_disabled_info/message'),
            width: '100%',
            style: {
                borderTop: '1px solid #a4b5c0 !important',
            },
            hidden: true,
        });

        this.items = [
            this.createGridPanel(),
        ];
        this.dockedItems = [
            {
                xtype: 'container',
                dock: 'top',
                layout: 'vbox',
                items: [
                    {
                        xtype: 'toolbar',
                        width: '100%',
                        ui: 'shopware-ui is--viison-common--toolbar',
                        padding: 12,
                        border: false,
                        items: this.getTopToolbarItems(),
                    },
                    this.notStockManagedInfo,
                ],
            },
        ];

        this.callParent(arguments);
    },

    /**
     * @return Ext.grid.Panel
     */
    createGridPanel: function () {
        this.gridPanel = Ext.create('Ext.grid.Panel', {
            cls: 'viison-common--grid has--vertical-lines has--no-borders',
            border: true,
            autoScroll: true,
            viewConfig: {
                enableTextSelection: true,
            },
            store: this.store,
            columns: this.createColumns(),
            dockedItems: [
                {
                    xtype: 'viison_common_pagination_toolbar-toolbar',
                    store: this.store,
                },
            ],
            plugins: [
                Ext.create('Ext.grid.plugin.RowEditing', {
                    clicksToEdit: 2,
                    autoCancel: true,
                    listeners: {
                        scope: this,
                        edit: function (editor, event) {
                            this.fireEvent('saveStockEntry', event.record);
                        },
                    },
                }),
            ],
        });

        return this.gridPanel;
    },

    /**
     * @return Ext.grid.column.Column[]
     */
    createColumns: function () {
        var showOrderTooltipSnippet = this.getViisonSnippet('action_column/show_order/tooltip');
        var showReturnShipmentTooltipSnippet = this.getViisonSnippet('action_column/show_return_shipment/tooltip');
        var showSupplierOrderTooltipSnippet = this.getViisonSnippet('action_column/show_supplier_order/tooltip');

        return [{
            header: this.getViisonSnippet('model/stock/type'),
            dataIndex: 'type',
            sortable: true,
            width: 80,
            scope: this,
            renderer: function (value, meta, record) {
                var snippetsForTypes = {
                    purchase: 'purchase',
                    sale: 'sale',
                    return: 'return',
                    stocktake: 'stocktake',
                    manual: 'manual',
                    incoming: 'incoming',
                    outgoing: 'outgoing',
                    relocation: 'relocation',
                    initialization: 'initialization',
                    'write-off': 'write_off',
                };
                var snippetName = snippetsForTypes[value] || 'unknown';
                var localizedTypeName = this.getViisonSnippet('store/stock_type/' + snippetName);

                if (record.get('correction')) {
                    localizedTypeName += this.getViisonSnippet('store/stock_type/correction_post_fix');
                }

                return localizedTypeName;
            },
        }, {
            header: this.getViisonSnippet('model/stock/article_number'),
            dataIndex: 'articleNumber',
            sortable: true,
            width: 100,
        }, {
            header: this.getViisonSnippet('model/stock/article_name'),
            dataIndex: 'articleName',
            sortable: true,
            flex: 1,
        }, {
            header: this.getViisonSnippet('model/stock/change_amount'),
            dataIndex: 'changeAmount',
            sortable: true,
            width: 65,
            align: 'right',
        }, {
            header: this.getViisonSnippet('model/stock/old_instock_quantity'),
            dataIndex: 'oldStock',
            sortable: true,
            width: 70,
            align: 'right',
        }, {
            header: this.getViisonSnippet('model/stock/new_instock_quantity'),
            dataIndex: 'newStock',
            sortable: true,
            width: 70,
            align: 'right',
        }, {
            xtype: 'numbercolumn',
            header: ViisonPickwarePurchasePriceHelper.purchasePriceLabel(),
            dataIndex: 'purchasePrice',
            sortable: true,
            renderer: ViisonCurrencyFormatter.renderer,
            width: 80,
            align: 'right',
            editor: {
                xtype: 'numberfield',
                allowBlank: false,
                decimalSeparator: ',',
                decimalPrecision: 2,
                submitLocaleSeparator: false,
                currencySymbol: ViisonCurrencyFormatter.currencySymbol,
            },
        }, {
            header: this.getViisonSnippet('model/stock/changed_bin_locations'),
            dataIndex: 'changedBinLocations',
            sortable: false,
            width: 140,
            renderer: function (value, metaData) {
                if (value === null || (Ext.isObject(value) && Ext.Object.getSize(value) === 0)) {
                    return '';
                }

                // Prepare tooltip and cell value
                var tooltipRows = [];
                var valueElements = [];
                Ext.Object.each(value, function (binLocationCode, stockChange) {
                    stockChange = (stockChange > 0) ? ('+' + stockChange) : stockChange;
                    tooltipRows.push('<b>' + binLocationCode + '</b>: ' + stockChange);
                    valueElements.push(binLocationCode + ' (' + stockChange + ')');
                });

                metaData.tdAttr = 'data-qtip="' + tooltipRows.join('<br/>') + '"';
                value = valueElements.join(', ');

                return value;
            },
        }, {
            header: this.getViisonSnippet('model/stock/username'),
            dataIndex: 'username',
            sortable: true,
            width: 100,
        }, {
            header: this.getViisonSnippet('model/stock/created'),
            xtype: 'datecolumn',
            dataIndex: 'created',
            sortable: true,
            width: 120,
            format: 'd.m.Y H:i:s',
        }, {
            header: this.getViisonSnippet('model/stock/comment'),
            dataIndex: 'comment',
            sortable: false,
            flex: 1,
            renderer: function (value, metaData) {
                if (value === null) {
                    return '';
                }
                metaData.tdAttr = 'data-qtip="' + value + '"';

                return value;
            },
        }, {
            xtype: 'actioncolumn',
            align: 'center',
            width: 60,
            items: [{
                iconCls: 'sprite-sticky-notes-pin',
                getClass: function (value, meta, record) {
                    // Determine tooltip or hide icon, if no action is available
                    if (record.get('returnShipmentId')) {
                        // Check returnShipmentId first, because if it is set, an orderId will be set too
                        this.items[0].tooltip = showReturnShipmentTooltipSnippet;
                    } else if (record.get('orderId')) {
                        this.items[0].tooltip = showOrderTooltipSnippet;
                    } else if (record.get('supplierOrderId')) {
                        this.items[0].tooltip = showSupplierOrderTooltipSnippet;
                    } else {
                        return 'x-hide-visibility';
                    }

                    return undefined;
                },
                handler: function (view, rowIndex) {
                    // Determine event name
                    var record = view.getStore().getAt(rowIndex);
                    var eventName;
                    if (record.get('returnShipmentId')) {
                        Shopware.app.Application.addSubApplication({
                            name: 'Shopware.apps.ViisonPickwareERPReturnShipment',
                            action: 'detail',
                            returnShipmentId: record.get('returnShipmentId'),
                        });
                    } else if (record.get('orderId')) {
                        eventName = 'showOrder';
                    } else if (record.get('supplierOrderId')) {
                        eventName = 'showSupplierOrder';
                    } else {
                        return;
                    }

                    this.fireEvent(eventName, record);
                }.bind(this),
            }, {
                iconCls: 'sprite-pencil',
                tooltip: this.getViisonSnippet('action_column/edit_purchase_price/tooltip'),
                scope: this,
                handler: function (view, rowIndex) {
                    // Activate row editing for column 6 (purchase price)
                    this.gridPanel.plugins[0].startEdit(rowIndex, 6);
                },
            }],
        }];
    },

    /**
     * @return Ext.Component[]
     */
    getTopToolbarItems: function () {
        // Create a combobox for the stock type selection and select its first value
        var stockTypeSelection = Ext.create('Ext.form.field.ComboBox', {
            name: 'type',
            valueField: 'name',
            displayField: 'title',
            queryMode: 'local',
            store: this.stockTypeStore,
            hideLabel: true,
            editable: false,
            allowBlank: false,
            forceSelection: true,
            listeners: {
                scope: this,
                change: function (combo, newValue, oldValue) {
                    if (newValue !== oldValue) {
                        this.fireEvent('typeChanged', newValue);
                    }
                },
            },
        });
        stockTypeSelection.select(this.stockTypeStore.getAt(0));

        return [
            {
                xtype: 'label',
                cls: 'is--viison-common--label',
                text: ViisonCommonApp.getSnippet(
                    'field_label',
                    'backend/viison_common_variant_combo_box/main'
                ) + ':',
                margin: '0 10 6 0',
            }, {
                xtype: 'viison_common_variant_combo_box-combo_box',
                name: 'variantSelection',
                width: 200,
                hideLabel: true,
            },
            '->',
            {
                xtype: 'tbspacer',
                width: 25,
            }, {
                xtype: 'label',
                cls: 'is--viison-common--label',
                text: ViisonCommonApp.getSnippet('warehouse_combobox/field_label', 'backend/viison_pickware_erp_warehouse_management/main') + ':',
                margin: '0 10 6 0',
            }, {
                xtype: 'viison_pickware_erp_warehouse_management-warehouse_combo_box',
                name: 'warehouseSelection',
                width: 200,
                hideLabel: true,
            }, {
                xtype: 'tbspacer',
                width: 25,
            }, {
                xtype: 'label',
                cls: 'is--viison-common--label',
                text: this.getViisonSnippet('toolbar/filter/type/label') + ':',
                margin: '0 10 6 0',
            },
            stockTypeSelection,
        ];
    },

});
