// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleStockFieldSet.controller.Detail', {

    override: 'Shopware.apps.Article.controller.Detail',

    /**
     * @Override
     */
    init: function () {
        this.callParent(arguments);
        this.control({
            'viison_pickware_erp_article_stock_field_set-stock_grid': {
                viisonPickwareERPSaveAvailableStock: this.onViisonPickwareERPSaveAvailableStock,
            },
        });
    },

    /**
     * @Override
     */
    onSaveArticle: function (win, article, options) {
        // Replace the options' callback to be able to update the stock field set
        var mainWindow = win || this.getMainWindow();
        var existingCallback = (Ext.isObject(options) && Ext.isFunction(options.callback)) ? options.callback : Ext.emptyFn;
        options.callback = function (savedArticle, success) {
            if (savedArticle && success) {
                // Update the stock field set
                var mainDetail = savedArticle.getMainDetail().first();
                var stockFieldSet = mainWindow.down('viison_pickware_erp_article_stock_field_set-stock_field_set');
                stockFieldSet.reloadData(mainDetail);
            }

            // Call the original callback if present
            existingCallback(savedArticle, success);
        };

        if (win) {
            // Make sure the available stock editing is cancelled before saving
            win.down('viison_pickware_erp_article_stock_field_set-stock_grid').cancelEditing();
        }

        this.callParent([win, article, options]);
    },

    /**
     * @Override
     */
    reconfigureAssociationComponents: function () {
        // Find the stock fieldset and set the flag to ignore changes of 'not relevant for stock manager', since this
        // method is called when chaning the article in split view and calling parent will assign new values to the fieldset
        var stockFieldSet = this.getMainWindow().down('viison_pickware_erp_article_stock_field_set-stock_field_set');
        stockFieldSet.blockOnChangeListenerOfNotStockManagedField = true;
        this.callParent(arguments);
        stockFieldSet.blockOnChangeListenerOfNotStockManagedField = false;

        // Manually reload the fieldset data to configure the fields correctly for the new article
        stockFieldSet.reloadData(this.getMainWindow().article.getMainDetail().first());
    },

    /**
     * Updates the available stock (instock) of a given article detail and rerenders
     * necessary parts of the detail / variants dialog. This method handels
     * the stock update for the main detail (article view) as well as for all
     * other details (variant view).
     */
    onViisonPickwareERPSaveAvailableStock: function (fieldSet, articleDetail, availableStock, oldAvailableInstock) {
        // Set available stock to new value
        this.viisonPickwareERPUpdateAvailableStock(fieldSet, articleDetail, availableStock);

        // Show loading mask
        this.getMainWindow().setLoading(true);

        Ext.Ajax.request({
            url: ViisonCommonApp.assembleBackendUrl('ViisonPickwareERPArticleStock/saveAvailableStock'),
            method: 'POST',
            params: {
                articleDetailId: articleDetail.get('id'),
                availableStock: availableStock,
            },
            scope: this,
            callback: function (options, requestSuccess, response) {
                var json = response.responseText && Ext.JSON.decode(response.responseText, true);
                var success = requestSuccess && json && json.success;

                if (success) {
                    Shopware.Notification.createGrowlMessage(
                        ViisonCommonApp.getSnippet(
                            'action/save_available_stock/success/title',
                            'backend/viison_pickware_erp_article_stock_field_set/main'
                        ),
                        Ext.String.format(
                            ViisonCommonApp.getSnippet(
                                'action/save_available_stock/success/message',
                                'backend/viison_pickware_erp_article_stock_field_set/main'
                            ),
                            articleDetail.get('number'),
                            articleDetail.get('id'),
                            oldAvailableInstock,
                            availableStock
                        ),
                        'Pickware ERP'
                    );

                    // Refresh article list in order to updated the displayed
                    // available stock value there as well
                    this.refreshArticleList();
                } else {
                    Shopware.Notification.createGrowlMessage(
                        ViisonCommonApp.getSnippet(
                            'action/save_available_stock/failure/title',
                            'backend/viison_pickware_erp_article_stock_field_set/main'
                        ),
                        Ext.String.format(
                            ViisonCommonApp.getSnippet(
                                'action/save_available_stock/failure/message',
                                'backend/viison_pickware_erp_article_stock_field_set/main'
                            ),
                            articleDetail.get('number'),
                            articleDetail.get('id')
                        ),
                        'Pickware ERP'
                    );

                    // Reset available stock to old value
                    this.viisonPickwareERPUpdateAvailableStock(fieldSet, articleDetail, oldAvailableInstock);
                }

                // Hide loading mask
                this.getMainWindow().setLoading(false);
            },
        });
    },

    /**
     * Updates the displayed available stock of a given stock field set.
     */
    viisonPickwareERPUpdateAvailableStock: function (fieldSet, articleDetail, availableStock) {
        articleDetail.set('inStock', availableStock);
        articleDetail.set('pickwareReservedStock', 0); // Passed value is ignored and calculated instead, we just use the setter to trigger that calculation
        fieldSet.reloadData(articleDetail);
    },

});
