// Copyright (c) Pickware GmbH. All rights reserved.
// This file is part of software that is released under a proprietary license.
// You must not copy, modify, distribute, make publicly available, or execute
// its contents or parts thereof without express permission by the copyright
// holder, unless otherwise permitted by law.

Ext.define('Shopware.apps.ViisonPickwareERPArticleStockFieldSet.view.StockFieldSet', {

    extend: 'Ext.form.FieldSet',
    alias: 'widget.viison_pickware_erp_article_stock_field_set-stock_field_set',
    mixins: [
        'Shopware.apps.ViisonCommonApp.Mixin',
    ],
    viisonSnippetNamespace: 'backend/viison_pickware_erp_article_stock_field_set/main',
    layout: 'column',
    /**
     * A default width for the field labels.
     */
    labelWidth: 155,
    /**
     * The article detail, whose data the stock field set is editing.
     */
    articleDetail: null,
    /**
     * A flag controlling whether the 'onSale' checkox should be hidden.
     *
     * This is used for the variant detail form, since variants did not have the "lastStock" option prior to Shopware 5.4.
     */
    onSaleCheckboxHidden: false,
    /**
     * A flag that is used by 'reloadData()', to temporarily disable the 'change' listener of the not stock managed field.
     * Please notice, that this flag is initially set to true in order to disable the 'change' listener at all before the
     * data record is fully loaded.
     */
    blockOnChangeListenerOfNotStockManagedField: true,
    /**
     * The names used for the form fields to autofill the article data.
     */
    fieldNames: {
        physicalStock: 'pickwarePhysicalStockForSale',
        reservedStock: 'pickwareReservedStock',
        notStockManaged: 'pickwareStockManagementDisabled',
        availableStock: 'inStock',
        minimumStock: 'stockMin',
        onSale: 'lastStock',
    },
    /**
     * The field style applied on all stock fields.
     */
    stockFieldStyle: {
        color: '#475C6A',
        'font-size': '1.3em',
        'font-weight': 'bold',
        'text-shadow': '0 1px 0 #fff',
        'text-align': 'right',
    },

    /**
     * @Override
     */
    initComponent: function () {
        this.setTitle(this.getViisonSnippet('title'));
        this.items = [
            this.createLeftColumn(),
            this.createRightColumn(),
        ];

        this.callParent(arguments);
    },

    /**
     * Update the article detail this field set describes.
     *
     * Updates the fields with the data of the given article detail.
     * Also updates the visibility and active state of components based on whether the article
     * detail is marked "relevant for stock manager".
     *
     * @param Shopware.apps.Article.model.Detail articleDetail
     */
    reloadData: function (articleDetail) {
        this.blockOnChangeListenerOfNotStockManagedField = true;

        this.stockGrid.setArticleDetail(articleDetail);
        this.notStockManagedField.setValue(articleDetail.get('pickwareStockManagementDisabled'));
        this.minimumStockField.setValue(articleDetail.get('stockMin'));
        this.articleDetail = articleDetail;

        this.blockOnChangeListenerOfNotStockManagedField = false;
    },

    /**
     * @return Ext.container.Container
     */
    createLeftColumn: function () {
        this.stockGrid = Ext.create('Shopware.apps.ViisonPickwareERPArticleStockFieldSet.view.StockGrid', {
            columnWidth: 0.5,
            margin: '0 20 0 0',
            articleDetail: this.articleDetail,
        });

        return this.stockGrid;
    },

    /**
     * @return Ext.container.Container
     */
    createRightColumn: function () {
        // Create the fields
        this.minimumStockField = Ext.create('Ext.form.field.Number', {
            name: this.fieldNames.minimumStock,
            itemId: 'minimumStockField',
            fieldLabel: this.getViisonSnippet('field/minimum_stock/field_label'),
            labelWidth: this.labelWidth,
            decimalPrecision: 0,
            readOnly: true,
            disabled: true,
            helpText: this.getViisonSnippet('field/minimum_stock/tooltip'),
        });
        this.onSaleField = Ext.create('Ext.form.field.Checkbox', {
            name: this.fieldNames.onSale,
            fieldLabel: this.getViisonSnippet('field/on_sale/field_label'),
            labelWidth: this.labelWidth,
            boxLabel: this.getViisonSnippet('field/on_sale/checkbox_text'),
            inputValue: true,
            uncheckedValue: false,
            hidden: this.onSaleCheckboxHidden,
        });
        this.notStockManagedField = Ext.create('Ext.form.field.Checkbox', {
            name: this.fieldNames.notStockManaged,
            fieldLabel: this.getViisonSnippet('field/not_stock_managed/field_label'),
            labelWidth: this.labelWidth,
            boxLabel: this.getViisonSnippet('field/not_stock_managed/box_label'),
            helpText: this.getViisonSnippet('field/not_stock_managed/help_text'),
            inputValue: true,
            uncheckedValue: false,
        });

        // List on changes of the 'notStockManaged' fields to present an alert notifying about the consequences
        this.notStockManagedField.on('change', function (field, newValue) {
            if (this.blockOnChangeListenerOfNotStockManagedField || field.readOnly || newValue !== true) {
                return;
            }

            Ext.Msg.confirm(
                this.getViisonSnippet('field/not_stock_managed/change_alert/title'),
                this.getViisonSnippet('field/not_stock_managed/change_alert/message'),
                function (buttonId) {
                    if (buttonId !== 'yes') {
                        // Reset the checkbox
                        this.notStockManagedField.setValue(false);
                    }
                },
                this
            );
        }, this);

        return {
            xtype: 'container',
            margin: '34 0 0 0',
            columnWidth: 0.5,
            layout: 'anchor',
            defaults: {
                anchor: '100%',
                labelWidth: 155,
            },
            items: [
                this.minimumStockField,
                this.onSaleField,
                this.notStockManagedField,
            ],
        };
    },

});
